from abc import abstractmethod

from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

import cars.settings
from cars.admin.permissions import AdminPermissionCode, AdminPermissionFactory
from cars.core.util import import_class
from .base import CallcenterInternalAPIView


class CarCommandView(CallcenterInternalAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    telematics_proxy = import_class(cars.settings.TELEMATICS['proxy_class']).from_settings()

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        if self.car is None:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        self.do_command(self.car)
        return Response()

    @abstractmethod
    def do_command(self, car):
        raise NotImplementedError


class OpenCarCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.open(car.imei)

class CloseCarCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.close(car.imei)

class UnlockHoodCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.unlock_hood(car.imei)

class LockHoodCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.lock_hood(car.imei)

class StartLeaseCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.start_lease(car.imei)

class EndLeaseCommandView(CarCommandView):

    def do_command(self, car):
        self.telematics_proxy.end_lease(car.imei)
