from rest_framework.response import Response
from rest_framework.status import HTTP_400_BAD_REQUEST

from cars.admin.permissions import AdminPermissionCode, AdminPermissionFactory
from cars.callcenter.serializers import (
    CallcenterCarSerializer,
    UpdateFuelCardNumberViewArgumentsSerializer
)
from cars.carsharing.models import Car
from cars.eka.core.processor import FuelCardProcessor
from cars.eka.models import FuelCardActivation
from .base import CallcenterInternalAPIView


class UpdateFuelCardNumberView(CallcenterInternalAPIView):

    arguments_serializer_class = UpdateFuelCardNumberViewArgumentsSerializer

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        if self.car is None:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        processor = FuelCardProcessor.from_order(
            order=self.order,
            operator=request.user,
            source=FuelCardActivation.Source.INTERNAL_CC.value,
        )
        processor.change_card_number(request.arguments['number'])

        updated_car = Car.objects.get(id=self.car.id)

        return Response(CallcenterCarSerializer(updated_car).data)


class FuelCardActivateView(CallcenterInternalAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        if self.order is None:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        processor = FuelCardProcessor.from_order(
            order=self.order,
            operator=request.user,
            source=FuelCardActivation.Source.INTERNAL_CC.value,
        )
        processor.activate_fuel_card()

        return Response(CallcenterCarSerializer(self.car).data)


class FuelCardBlockView(CallcenterInternalAPIView):

    permission_classes = [AdminPermissionFactory.build(AdminPermissionCode.OPERATE_CARS)]

    def do_post(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        if self.order is None:
            return Response(
                {
                    'error': 'order.missing',
                },
                status=HTTP_400_BAD_REQUEST,
            )

        processor = FuelCardProcessor.from_order(
            order=self.order,
            operator=request.user,
            source=FuelCardActivation.Source.INTERNAL_CC.value,
        )
        processor.block_fuel_card()

        return Response(CallcenterCarSerializer(self.car).data)
