import logging
import time

from django.db import transaction
from django.utils import timezone

from cars.carsharing.models.car_document import CarDocumentAssignment, CarDocument
from ..models.car import Car


LOGGER = logging.getLogger(__name__)


class CarUpdater:

    ALLOWED_STATUS_TRANSITIONS = {
        Car.Status.ACCEPTANCE: {
            Car.Status.AVAILABLE,
            Car.Status.PARKING,
            Car.Status.RIDE,
        },
        Car.Status.AVAILABLE: {
            Car.Status.CLEANING,
            Car.Status.FUELING,
            Car.Status.RESERVATION,
            Car.Status.SERVICE,
        },
        Car.Status.FUELING: {
            Car.Status.AVAILABLE,
        },
        Car.Status.NEW: {
            Car.Status.AVAILABLE,
            Car.Status.SERVICE,
        },
        Car.Status.PARKING: {
            Car.Status.AVAILABLE,
            Car.Status.RIDE,
        },
        Car.Status.RESERVATION: {
            Car.Status.ACCEPTANCE,
            Car.Status.AVAILABLE,
            Car.Status.PARKING,
            Car.Status.RIDE,
            Car.Status.RESERVATION_PAID,
        },
        Car.Status.RESERVATION_PAID: {
            Car.Status.ACCEPTANCE,
            Car.Status.AVAILABLE,
            Car.Status.PARKING,
            Car.Status.RIDE,
        },
        Car.Status.RIDE: {
            Car.Status.AVAILABLE,
            Car.Status.PARKING,
        },
        Car.Status.SERVICE: {
            Car.Status.AVAILABLE,
        },
        Car.Status.CLEANING: {
            Car.Status.AVAILABLE,
            Car.Status.SERVICE,
        }
    }

    class BadStatusError(Exception):
        pass

    def __init__(self, car):
        self._car = car

    def assign_document(self, document, assigner_user, history_manager=None, force_detach=True):
        with transaction.atomic():
            # if this document is attached to something else, detach it from there
            current_document_assignment = (
                CarDocumentAssignment.objects
                .filter(
                    document=document,
                    unassigned_at__isnull=True,
                )
                .first()
            )
            if current_document_assignment is not None:
                self._complete_document_assignment(current_document_assignment, assigner_user)
                if history_manager:
                    history_manager.remove_entry(current_document_assignment, str(assigner_user.id))

            # if this car has an assigned document of this type, detach it
            current_document_assignment = (
                CarDocumentAssignment.objects
                .filter(
                    car=self._car,
                    document__type=document.type,
                    unassigned_at__isnull=True,
                )
                .first()
            )
            if current_document_assignment is not None and force_detach:
                self._complete_document_assignment(current_document_assignment, assigner_user)
                if history_manager:
                    history_manager.remove_entry(current_document_assignment, str(assigner_user.id))

            new_document_assignment = CarDocumentAssignment(
                document=document,
                car=self._car,
                assigned_at=timezone.now(),
                assigned_by=assigner_user,
            )
            new_document_assignment.save()
            if history_manager:
                history_manager.add_entry(new_document_assignment, str(assigner_user.id))

            if document.type == CarDocument.Type.CAR_HARDWARE_VEGA.value:
                imei = document.get_impl().imei
                if imei is not None:
                    self._car.imei = document.get_impl().imei
                    self._car.updated_at = timezone.now()
                    self._car.update_timestamp = time.time()
                    self._car.save()

    def unassign_document(self, document_assignment, operator, history_manager=None):
        with transaction.atomic():
            if document_assignment.document.type == CarDocument.Type.CAR_HARDWARE_VEGA.value:
                document_assignment.car.imei = None
                document_assignment.car.updated_at = timezone.now()

            document_assignment.car.update_timestamp = time.time()
            document_assignment.car.save()

            document_assignment.unassigned_at = timezone.now()
            document_assignment.unassigned_by = operator
            document_assignment.save()

            if history_manager:
                history_manager.remove_entry(document_assignment, str(operator.id))

    def update_car_basic_information(self, registry_document):
        if registry_document.contract_id is not None:
            self._car.contract_id = registry_document.contract_id
        if registry_document.registration_id is not None:
            self._car.registration_id = registry_document.registration_id
        if registry_document.registration_date is not None:
            self._car.registration_date = registry_document.registration_date
        if registry_document.fuel_card_number is not None:
            self._car.fuel_card_number = registry_document.fuel_card_number
        if registry_document.number is not None:
            self._car.number = registry_document.number
        if registry_document.imei is not None:
            self._car.imei = registry_document.imei
        self._car.updated_at = timezone.now()
        self._car.update_timestamp = time.time()
        self._car.save()

    def update_status(self, status, allowed_source_statuses=None):
        with transaction.atomic():
            car = Car.objects.select_for_update().get(id=self._car.id)
            car_status = car.get_status()

            if car_status is status:
                LOGGER.info('car %s is already %s', car.id, status)
                return

            if allowed_source_statuses and car_status not in allowed_source_statuses:
                raise self.BadStatusError

            allowed_transitions = self.ALLOWED_STATUS_TRANSITIONS.get(car_status)
            if allowed_transitions is None:
                LOGGER.error('status %s not found in transitions graph', status)
                raise self.BadStatusError

            if status not in allowed_transitions:
                raise self.BadStatusError

            LOGGER.info('updating car %s status from %s to %s', car.id, car_status, status)
            car.status = status.value
            car.updated_at = timezone.now()
            car.update_timestamp = time.time()
            try:
                car.save()
            except Exception:
                car.refresh_from_db()
                raise

    def update_fuel_card_number(self, fuel_card_number):
        self._car.fuel_card_number = fuel_card_number
        self._car.updated_at = timezone.now()
        self._car.update_timestamp = time.time()
        self._car.save()

    def _complete_document_assignment(self, document_assignment, operator):
        document_assignment.unassigned_at = timezone.now()
        document_assignment.unassigned_by = operator
        document_assignment.save()
