import enum
import re

from cars.carsharing.models.car import Car
from cars.carsharing.models.car_registry_document import CarRegistryDocument
from cars.core.util import import_class
import cars.settings as settings


class BadFilenameError(ValueError):

    def __init__(self, type_, filename):
        self._type = type_
        self._filename = filename
        super().__init__(self)

    def __str__(self):
        return 'bad filename for {} document: {}'.format(
            self._type.value,
            self._filename
        )


class DocumentFilesManager:

    class DocumentFileType(enum.Enum):
        TRANSFER_RECEIPT = 'transfer_receipt'
        REGISTRATION = 'registration'  # sts
        OSAGO = 'osago'

    def __init__(self, mds_client, mds_bucket_name):
        self._mds_client = mds_client
        self._mds_bucket_name = mds_bucket_name

    @classmethod
    def from_settings(cls):
        return cls(
            mds_bucket_name=settings.CAR_DOCUMENTS['mds']['mds_bucket_name'],
            mds_client=import_class(settings.MDS['client_class']).from_settings(),
        )

    def save_document(self, *, filename, data, type_):
        document = self._find_document_by_type(filename, type_)
        mds_key = self._get_document_key(filename, type_)
        self._upload_to_mds(key=mds_key, data=data)
        attrname = '{}_mds_key'.format(type_.value)
        setattr(document, attrname, mds_key)
        document.save()

    def _upload_to_mds(self, *, key, data):
        self._mds_client.put_object(
            key=key,
            bucket=self._mds_bucket_name,
            body=data,
        )

    def _get_document_key(self, filename, type_):
        return '{}/{}'.format(type_.value, filename)

    def _find_document_by_type(self, filename, type_):
        if type_ == self.DocumentFileType.OSAGO:
            m = re.match(r'OSAGO_(\d+)', filename)
            if m is None:
                raise BadFilenameError(type_, filename)
            osago_number = 'XXX{}'.format(m.group(1))
            return CarRegistryDocument.objects.get(
                osago_number=osago_number
            )

        elif type_ == self.DocumentFileType.TRANSFER_RECEIPT:
            m = re.match('APP_([а-яА-Я\d]+)', filename)
            if m is None:
                raise BadFilenameError(type_, filename)
            car_number = m.group(1).lower()
            return CarRegistryDocument.objects.get(
                number=car_number,
            )

        elif type_ == self.DocumentFileType.REGISTRATION:
            m = re.match('STS_(\d+)', filename)
            if m is None:
                raise BadFilenameError(type_, filename)
            registration_id = m.group(1)
            return CarRegistryDocument.objects.get(
                registration_id=registration_id
            )

        else:
            raise TypeError('unknown document file type `{}`'.format(type_))
