import datetime
import decimal
import json
import logging
import threading
import time
from collections import defaultdict

import pandas
import pytz
from django.db import transaction
from django.utils import timezone

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.models import Car
from cars.core.history import HistoryManager
from cars.service_app.core.assembly_manager import AssemblyManager
from cars.service_app.models.history import DriveCarDocumentHistory, DriveCarDocumentAssignmentHistory
from cars.carsharing.models.car_document import CarDocument, CarDocumentAssignment


LOGGER = logging.getLogger(__name__)


class CarInsuranceManager:

    EXCEL_FIELDS_TO_INNER_REPR_MATCHING = {
        'agreementpartnernum': 'agreement_partner_number',
        'agreementnum': 'agreement_number',
        'поминутное': 'per_minute_cost',
        'дата начала страхования': 'valid_from',
        'дата окончания страхования': 'valid_until',
    }

    def __init__(self, operator_user):
        self._user = operator_user

    def bulk_add_insurance_from_df(self, df):
        t = threading.Thread(
            target=self._bulk_add_insurance_from_df,
            args=(df, ),
            daemon=True,
        )
        t.start()
        if cars.settings.CARSHARING['registry_manager']['join_threads']:
            LOGGER.info('waiting for the thread to be joined')
            t.join()

    def _bulk_add_insurance_from_df(self, df):
        existing_insurances = self._get_existing_insurances_mapping()

        df = df.fillna('')
        for index, row in df.iterrows():
            row_lower = self._get_normalized_dict_from_df_row(row)
            try:
                new_policy = self._construct_insurance_object_from_dict(row_lower)
            except Exception:
                continue
            if new_policy is None:
                continue
            new_policy['per_minute_cost'] = str(new_policy['per_minute_cost'])

            car_id = new_policy['car']
            del new_policy['car']
            existing_policies = existing_insurances[car_id]

            old_policy = None
            have_difference = False
            if existing_policies:
                for policy_document in existing_policies:
                    try:
                        policy = json.loads(policy_document.blob)
                    except Exception:
                        continue
                    else:
                        if 'valid_from' not in policy:
                            policy['valid_from'] = None
                        if 'valid_until' not in policy:
                            policy['valid_until'] = None

                    is_twin_by_numbers = policy['agreement_partner_number'] == new_policy['agreement_partner_number'] and policy['agreement_number'] == new_policy['agreement_number']
                    is_twin_by_dates = policy['valid_from'] == new_policy['valid_from'] and policy['valid_until'] == new_policy['valid_until']
                    if is_twin_by_dates or is_twin_by_numbers:
                        old_policy = policy_document
                        have_difference = (
                            policy['per_minute_cost'] != new_policy['per_minute_cost'] or
                            policy['base_cost'] != new_policy['base_cost'] or
                            policy['valid_from'] != new_policy['valid_from'] or
                            policy['valid_until'] != new_policy['valid_until']
                        )
                        break

            old_id = None
            if old_policy is not None:
                old_id = old_policy.id

            if have_difference or old_policy is None:
                time.sleep(1)
                try:
                    with transaction.atomic():
                        new_policy_document = CarDocument(
                            added_at=timezone.now(),
                            added_by=self._user,
                            type=CarDocument.Type.CAR_INSURANCE_POLICY.value,
                            blob=json.dumps(new_policy),
                        )
                        new_policy_document.save()

                        # Write history
                        HistoryManager(DriveCarDocumentHistory).add_entry(new_policy_document, str(self._user.id))

                        updater = CarUpdater(Car.objects.get(id=car_id))
                        if old_policy is not None:
                            old_assignment = (
                                CarDocumentAssignment.objects
                                .filter(
                                    unassigned_at__isnull=True,
                                    document=old_policy
                                )
                            ).first()
                            if old_assignment:
                                updater.unassign_document(
                                    old_assignment,
                                    self._user,
                                    history_manager=HistoryManager(DriveCarDocumentAssignmentHistory),
                                )
                                HistoryManager(DriveCarDocumentHistory).remove_entry(old_policy, str(self._user.id))
                                old_policy.delete()

                        updater.assign_document(
                            new_policy_document,
                            assigner_user=self._user,
                            history_manager=HistoryManager(DriveCarDocumentAssignmentHistory),
                            force_detach=False
                        )
                except Exception:
                    LOGGER.exception(
                        'unable to save new insurance policy for car=%s',
                        str(car_id)
                    )
                else:
                    existing_insurances[car_id].append(new_policy_document)
                    new_list = []
                    for obj in existing_insurances[car_id]:
                        if obj.id != old_id:
                            new_list.append(obj)
                    existing_insurances[car_id] = new_list

    def _get_existing_insurances_mapping(self):
        mapping = defaultdict(list)
        assignments = (
            CarDocumentAssignment.objects
            .filter(
                unassigned_at__isnull=True,
                document__type='car_insurance_policy'
            )
            .select_related('document')
        )

        for assignment in assignments:
            car_id = str(assignment.car_id)
            insurance = assignment.document
            mapping[car_id].append(insurance)

        return mapping

    def _policies_differ(self, old_policy, new_policy):
        if old_policy is None:
            return new_policy is not None
        if new_policy is None:
            return False

        if old_policy.agreement_number != new_policy.agreement_number:
            return True
        if old_policy.agreement_partner_number != new_policy.agreement_partner_number:
            return True
        if old_policy.base_cost != new_policy.base_cost:
            return True
        if old_policy.per_minute_cost != new_policy.per_minute_cost:
            return True
        return False

    def _get_normalized_dict_from_df_row(self, row):
        row_lower = {}
        for key, value in row.items():
            if isinstance(value, pandas.Timestamp):
                value = timezone.make_aware(value.to_pydatetime(), pytz.UTC)
            row_lower[key.lower()] = value

        return row_lower

    def _construct_insurance_object_from_dict(self, row):
        car_insurance_kwargs = {}
        for k, v in self.EXCEL_FIELDS_TO_INNER_REPR_MATCHING.items():
            if k not in row:
                return None
            car_insurance_kwargs[v] = row[k]
            if v in ('valid_from', 'valid_until'):
                year = row[k].year
                month = row[k].month
                day = row[k].day
                validity_date = (
                    datetime.datetime(
                        year=year,
                        month=month,
                        day=day, tzinfo=pytz.UTC
                    )
                ) - datetime.timedelta(hours=3)
                car_insurance_kwargs[v] = validity_date.isoformat()

        car = Car.objects.filter(vin=row.get('vin', '').strip()).first()
        if car is None:
            return None
        car_insurance_kwargs['car'] = str(car.id)

        # According to https://st.yandex-team.ru/DRIVEBACK-44
        manufacturer = car.model.manufacturer
        car_insurance_kwargs['base_cost'] = '0.35'

        car_insurance_kwargs['agreement_partner_number'] = (
            car_insurance_kwargs['agreement_partner_number'].strip()
        )
        car_insurance_kwargs['agreement_number'] = (
            car_insurance_kwargs['agreement_number'].strip()
        )

        return car_insurance_kwargs
