import os

import shapely.geometry
import shapely.wkt


class ParkingAreaChecker:

    class Error(Exception):
        pass

    class ForbiddenLocationError(Error):
        def __init__(self):
            super().__init__('parking_area.location.forbidden')

    class UnknownLocationError(Error):
        def __init__(self):
            super().__init__('parking_area.location.unknown')

    def __init__(self, parking_area_effective, parking_area_visible):
        self._parking_area_effective = parking_area_effective
        self._parking_area_visible = parking_area_visible

    @classmethod
    def from_settings(cls):
        carsharing_root_path = os.path.dirname(os.path.dirname(__file__))
        resources_path = os.path.join(carsharing_root_path, 'core', 'resources')
        parking_area_effective_wkt_path = os.path.join(resources_path, 'parking_area_effective.wkt')
        parking_area_visible_wkt_path = os.path.join(resources_path, 'parking_area_visible.wkt')

        parking_area_effective = cls._load_shape(parking_area_effective_wkt_path)
        parking_area_visible = cls._load_shape(parking_area_visible_wkt_path)

        return cls(
            parking_area_effective=parking_area_effective,
            parking_area_visible=parking_area_visible,
        )

    @classmethod
    def _load_shape(cls, path):
        with open(path) as f:
            shape = shapely.wkt.load(f)
        return shape

    def check(self, location):
        if location is None:
            raise self.UnknownLocationError

        point = shapely.geometry.Point(location.lon, location.lat)
        if not self._parking_area_effective.contains(point):
            raise self.ForbiddenLocationError

    def get_parking_area_geojson(self):
        return shapely.geometry.mapping(self._parking_area_visible)
