import datetime
import logging
import threading
import time

from cars.orders.models import OrderItem

from cars.carsharing.models import CarsharingReservation
from django.db import connections

import pytz

import cars.settings


LOGGER = logging.getLogger(__name__)


class CarRecentlyCleanerGetter:
    """
    Determines which cars were last cleaned.
    """

    CACHE_TTL = 600.0

    def __init__(self):
        self._cars = set()
        self._last_update = 0

        self._lock = threading.Lock()
        self._update_in_progress = False

    def _fetch_last_cleaning_times(self):
        try:
            cursor = connections[cars.settings.DB_RO_ID].cursor()
            cursor.execute(
                "select object_id, max(timestamp) from car_tags_history where tag_name='cleaning'"
                " and action='remove' group by object_id;"
            )
            last_cleaning_times = cursor.fetchall()
        except Exception:
            LOGGER.exception('unable to fetch last cleaning times')
            last_cleaning_times = []

        result = {}
        for usage in last_cleaning_times:
            result[str(usage[0])] = int(usage[1])

        return result

    def _fetch_last_order_completions(self):
        result = {}
        try:
            last_usages = (
                CarsharingReservation.objects
                .using(cars.settings.DB_RO_ID)
                .only(
                    'car_id',
                    'order_item__started_at'
                )
                .order_by(
                    'car_id',
                    '-order_item__started_at',
                )
                .distinct('car_id')
                .select_related('order_item')
            )

            for usage in last_usages:
                result[str(usage.car_id)] = (
                    usage.order_item.started_at - datetime.datetime(year=1970, month=1, day=1, tzinfo=pytz.UTC)
                ).total_seconds()
        except Exception:
            LOGGER.exception('unable to fetch last order times')

        return result

    def _update_recently_cleaned_cars(self):
        result = set()
        last_cleaning_times = self._fetch_last_cleaning_times()
        last_order_times = self._fetch_last_order_completions()

        for car_id, last_cleaning_time in last_cleaning_times.items():
            last_order_time = last_order_times.get(car_id, 0)
            if last_cleaning_time > last_order_time:
                result.add(str(car_id))

        return result

    def _fetch_lists(self):
        new_clean_cars_list = self._update_recently_cleaned_cars()

        self._lock.acquire()

        self._cars = new_clean_cars_list
        self._update_in_progress = False
        self._last_update = time.time()

        self._lock.release()

    def _do_update(self):
        if self._update_in_progress:
            return

        self._lock.acquire()
        self._update_in_progress = True
        self._lock.release()

        t = threading.Thread(target=self._fetch_lists)
        t.start()

    def get_recently_cleaned_cars(self):
        timestamp = time.time()
        if timestamp - self._last_update > self.CACHE_TTL:
            self._do_update()

        self._lock.acquire()
        result = self._cars  # ensure no race
        self._lock.release()

        return result


RECENTLY_CLEANED_CARS_GETTER = CarRecentlyCleanerGetter()


class CarPropsCollector:

    def __init__(self):
        self._recently_cleaned_cars = {}

    def get_props(self, car_list):
        self._recently_cleaned_cars = RECENTLY_CLEANED_CARS_GETTER.get_recently_cleaned_cars()
        mapping = {}
        for car in car_list:
            props = []
            self._add_fuel_level_if_applies(car, props)
            self._add_recently_cleaned_if_applies(car, props)
            self._add_remote_cooling_if_applies(car, props)
            self._add_onboard_computer_if_applies(car, props)
            mapping[str(car.id)] = props
        return mapping

    def _add_fuel_level_if_applies(self, car, props):
        telematics_state = car.get_telematics_state()
        if not telematics_state:
            return
        try:
            if not telematics_state.fuel_level:
                return

            fuel_level = int(round(telematics_state.fuel_level))

            if fuel_level > 100:  # it really happens, looks like minor telematics issue
                fuel_level = 100
            if fuel_level < 0:
                fuel_level = 0

            item = {
                'name': 'Бак {}%'.format(fuel_level)
            }
            if fuel_level >= 90:
                item['image_url'] = (
                    'https://carsharing.s3.yandex.net/drive/static/tag-icons/gas-blue.png'
                )
                item['is_important'] = True
            else:
                item['image_url'] = (
                    'https://carsharing.s3.yandex.net/drive/static/tag-icons/gas-black.png'
                )
                item['is_important'] = False

            props.append(item)
        except Exception:
            # Shouldn't happen.
            # However, we can't afford falling on cars list stage
            LOGGER.exception('failure on forming props list; fuel level')

    def _add_remote_cooling_if_applies(self, car, props):  # pylint: disable=unused-argument
        item = {
            'image_url': (
                'https://carsharing.s3.yandex.net/drive/static/tag-icons/air-condition.png'
            ),
            'is_important': False,
            'name': 'Удаленное охлаждение'
        }
        props.append(item)

    def _add_onboard_computer_if_applies(self, car, props):
        if car.model.manufacturer in ('Porsche', 'Škoda', 'Audi'):
            return
        item = {
            'image_url': (
                'https://carsharing.s3.yandex.net/drive/static/tag-icons/computer-v2.png'
            ),
            'is_important': False,
            'name': 'Бортовой компьютер',
        }
        props.append(item)

    def _add_recently_cleaned_if_applies(self, car, props):
        if str(car.id) in self._recently_cleaned_cars:
            item = {
                'image_url': (
                    'https://carsharing.s3.yandex.net/drive/static/tag-icons/clean.png'
                ),
                'is_important': True,
                'name': 'Только что помыли'
            }
            props.append(item)
