import json
import logging
import threading
import time

import pandas
import pytz
from django.db import transaction
from django.utils import timezone

import cars.settings
from cars.carsharing.core.car_updater import CarUpdater
from cars.carsharing.models import CarRegistryDocument
from cars.carsharing.models.car_document import CarDocumentAssignment, CarDocument
from cars.core.history import HistoryManager
from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.service_app.core.assembly_manager import AssemblyManager
from cars.service_app.models.history import DriveCarDocumentHistory, DriveCarDocumentAssignmentHistory
from ..models import Car, CarModel


LOGGER = logging.getLogger(__name__)


class CarRegistryManager:

    EXCEL_FIELDS_TO_INNER_REPR_MATCHING = {
        'vin': 'vin',  # not a document field, only for car assignment
        'договор №': 'contract_id',
        'дата дс': 'contract_date',
        'марка': 'manufacturer',  # used only for car creation
        'модель': 'model',
        'грз': 'number',
        'стс': 'registration_id',
        'дата регистрации': 'registration_date',
        'дата передачи': 'transfer_date',
        'осаго №': 'osago_number',
        'осаго дата': 'osago_date',
        'птс.модель': 'mark_by_pts',
        'год выпуска': 'production_year',
        '№птс': 'pts_number',
        'противоугонная система': 'antitheft_system',
        'количество ключей': 'number_of_keys',
        'дата подачи документов в дептранс': 'deptrans_documents_date',
        'дата начала парковочного разрешения': 'parking_permit_start_date',
        'вега': 'imei',
        'топливная карта': 'fuel_card_number',  # requires additional processing
        'каско': 'casco_number',
    }

    DOCUMENT_MODEL_FIELDS = {
        'contract_id', 'contract_date', 'number', 'registration_id', 'registration_date',
        'transfer_date', 'osago_number', 'osago_date', 'mark_by_pts', 'production_year',
        'pts_number', 'antitheft_system', 'number_of_keys', 'deptrans_documents_date',
        'parking_permit_start_date', 'fuel_card_number', 'casco_number', 'imei',
    }

    REQUIRED_FIELDS = {
        'vin', 'марка', 'модель',
    }

    def __init__(self, operator_user):
        self.car_models = {}
        for cm in CarModel.objects.all():
            if cm.registry_manufacturer and cm.registry_model:
                key = (cm.registry_manufacturer.lower(), cm.registry_model.lower())
                self.car_models[key] = cm

        self._user = operator_user

    def bulk_add_and_update_cars_from_df(self, df):
        """
        Add missing and update existing cars from pandas.DataFrame-like object.
        """
        df = df.fillna('')

        addition_payload = []
        for index, row in df.iterrows():
            row_lower = {}
            for key, value in row.items():
                if isinstance(value, pandas.Timestamp):
                    value = timezone.make_aware(value.to_pydatetime(), pytz.UTC)
                row_lower[key.lower()] = value

            row_payload = self.construct_update_payload_from_dict(row_lower)
            if row_payload:
                addition_payload.append(row_payload)

        num_added = self.bulk_create_cars(addition_payload)
        self.bulk_update_cars_async(addition_payload)

        return num_added

    def construct_update_payload_from_dict(self, row):
        for field in self.REQUIRED_FIELDS:
            if field not in row or (not row[field]):
                return None

        payload = {}
        for key, value in row.items():
            if value == '':
                value = None  # we can't directly fillna to None
            if key.lower() in self.EXCEL_FIELDS_TO_INNER_REPR_MATCHING:
                field_name = self.EXCEL_FIELDS_TO_INNER_REPR_MATCHING[key.lower()]
                if isinstance(value, str):
                    value = value.strip()
                payload[field_name] = value

        return payload

    def bulk_create_cars(self, cars_):
        existing_vins = set()
        for car in Car.objects.all():
            existing_vins.add(car.vin)

        car_db_objs = []

        for car in cars_:
            if car['vin'] in existing_vins:
                # ignore existing
                continue

            model_obj = self.car_models.get(
                (car['manufacturer'].lower(), car['model'].lower(), )
            )
            if not model_obj:
                LOGGER.error('unable to match car model for car: %s', str(car))
                continue

            car_db_objs.append(
                Car(
                    model=model_obj,
                    vin=car['vin'],
                )
            )
            existing_vins.add(car['vin'])

        Car.objects.bulk_create(car_db_objs)

        if not cars.settings.IS_TESTS:
            self.assign_new_tags_async(car_db_objs)

        return len(car_db_objs)

    def bulk_update_cars_async(self, car_dicts):
        t = threading.Thread(
            target=self.bulk_update_cars,
            args=(car_dicts, ),
            daemon=True,
        )
        t.start()
        if cars.settings.CARSHARING['registry_manager']['join_threads']:
            LOGGER.info('waiting for the thread to be joined')
            t.join()

    def bulk_update_cars(self, cars):
        vin_to_car = self._get_vin_to_car_mapping()
        vin_to_latest_document = self._get_vin_to_latest_document_mapping()

        for car in cars:
            car['vin'] = car['vin'].strip()
            if car['vin'] not in vin_to_car:
                LOGGER.error('unknown vin: %s', car['vin'])
                continue
            latest_registry_document = vin_to_latest_document.get(car['vin'], CarRegistryDocument())
            new_registry_document = self._form_new_car_registry_document(car, latest_registry_document)
            if self._is_documents_differ(latest_registry_document, new_registry_document):
                updater = CarUpdater(vin_to_car[car['vin']])
                try:
                    with transaction.atomic():
                        new_registry_document.save()
                        document = CarDocument(
                            type=CarDocument.Type.CAR_REGISTRY_DOCUMENT.value,
                            car_registry_document=new_registry_document,
                            added_at=timezone.now(),
                            added_by=self._user,
                        )
                        AssemblyManager.update_blob(document)
                        HistoryManager(DriveCarDocumentHistory).add_entry(document, str(self._user.id))
                        document.save()
                        updater.assign_document(
                            document,
                            self._user,
                            history_manager=HistoryManager(DriveCarDocumentAssignmentHistory)
                        )
                        updater.update_car_basic_information(new_registry_document)
                except Exception:
                    LOGGER.exception('failed to update car %s', str(vin_to_car[car['vin']].id))

    def assign_new_tags_async(self, car_db_objs):
        t = threading.Thread(
            target=self.assign_new_tags,
            args=(car_db_objs, ),
            daemon=True,
        )
        t.start()
        if cars.settings.CARSHARING['registry_manager']['join_threads']:
            t.join()

    def assign_new_tags(self, car_list):
        time.sleep(300)
        client = SaasDriveAdminClient.from_settings()

        model_to_default_tags = {}
        for car_model in CarModel.objects.all():
            default_tags_str = car_model.default_tags
            if default_tags_str == '':
                default_tags_str = '[]'
            model_to_default_tags[car_model.code] = json.loads(default_tags_str)

        for car in car_list:
            model_code = car.model.code
            for default_tag_description in model_to_default_tags[model_code]:
                client.add_car_tag(
                    car_id=str(car.id),
                    tag_name=default_tag_description['tag_name'],
                    comment=default_tag_description['comment'],
                    priority=default_tag_description['priority'],
                )

    def _form_new_car_registry_document(self, car, latest_registry_document=None):
        document_kwargs = {}
        for field in self.DOCUMENT_MODEL_FIELDS:
            if isinstance(field, str):
                field = field.strip()
            value = car.get(field)
            if isinstance(value, str):
                value = value.strip()
            document_kwargs[field] = value

        if document_kwargs['fuel_card_number']:
            fuel_card_number = str(int(document_kwargs['fuel_card_number']))
            if fuel_card_number is not None and len(fuel_card_number) <= 6:
                fuel_card_number = '0' * (6 - len(fuel_card_number)) + fuel_card_number
                document_kwargs['fuel_card_number'] = '782555000000{}'.format(fuel_card_number)

        imei = document_kwargs['imei']
        if imei is not None:
            imei = imei.strip(' "')
            document_kwargs['imei'] = int(imei)

        number = document_kwargs['number']
        if number is not None:
            document_kwargs['number'] = number.lower()

        if latest_registry_document is not None:
            for field in self.DOCUMENT_MODEL_FIELDS:
                first_value = getattr(latest_registry_document, field, None)
                second_value = document_kwargs.get(field)
                if second_value is None and first_value is not None:
                    document_kwargs[field] = first_value

        return CarRegistryDocument(**document_kwargs)

    def _is_documents_differ(self, old_document, new_document):
        for field in self.DOCUMENT_MODEL_FIELDS:
            first_value = getattr(old_document, field, None)
            second_value = getattr(new_document, field, None)
            if first_value != second_value and second_value is not None:
                return True
        return False

    def _get_vin_to_car_mapping(self):
        mapping = {}
        for car in Car.objects.all():
            mapping[car.vin] = car
        return mapping

    def _get_vin_to_latest_document_mapping(self):
        mapping = {}
        latest_document_assignments = (
            CarDocumentAssignment.objects
            .filter(
                document__type=CarDocument.Type.CAR_REGISTRY_DOCUMENT.value,
                unassigned_at__isnull=True,
            )
            .select_related(
                'car',
                'document__car_registry_document',
            )
        )

        for assignment in latest_document_assignments:
            mapping[assignment.car.vin] = assignment.document.car_registry_document

        return mapping
