import logging
import os

from django.utils import timezone

import cars.settings
from cars.carsharing.models.car_tag_photo import CarsharingTagPhoto
from cars.core.images import Resizer, ImageProcessor
from cars.core.util import import_class


LOGGER = logging.getLogger(__name__)
MDSClientClass = import_class(cars.settings.MDS['client_class'])


class TagPhotoManager:

    def __init__(self, mds_bucket_name, images_access_host, mds_client, user, car):
        self._user = user
        self._car = car
        self._mds_client = mds_client

        self._mds_bucket_name = mds_bucket_name
        self._images_access_host = images_access_host

        resizer = Resizer(
            cars.settings.SERVICE_APP['tags']['photos']['thumbnail_greater_side']
        )
        self._image_processor = ImageProcessor(
            watermarker=None,
            resizer=resizer,
        )

    @classmethod
    def from_settings(cls, user, car):
        return cls(
            mds_bucket_name=cars.settings.SERVICE_APP['tags']['photos']['mds_bucket_name'],
            images_access_host=cars.settings.SERVICE_APP['tags']['photos']['images_access_host'],
            mds_client=MDSClientClass.from_settings(),
            user=user,
            car=car,
        )

    def upload_photo(self, photo_content, photo_id, type_, tags):
        key = self._get_photo_mds_key(type_, photo_id)

        # first upload the photo itself
        self._mds_client.put_object(
            bucket=self._mds_bucket_name,
            key=self._get_photo_mds_key(type_, photo_id),
            body=photo_content,
        )

        self._mds_client.put_object(
            bucket=self._mds_bucket_name,
            key=self._get_photo_thumbnail_mds_key(type_, photo_id),
            body=self._image_processor.process(photo_content),
        )

        # then create correponding db records

        result = []
        for tag in tags:
            photo = CarsharingTagPhoto(
                car=self._car,
                tag_id=tag,
                type=type_,

                submitted_by=self._user,
                submitted_at=timezone.now(),

                link=self._get_photo_url(key),
                thumbnail_link=self._get_photo_thumbnail_url(key),
            )
            photo.save()
            result.append(photo)

        return result

    def _get_photo_mds_key(self, type_, photo_id):
        return os.path.join(
            str(self._car.id),
            type_,
            photo_id
        )

    def _get_photo_url(self, key):
        return os.path.join(
            self._images_access_host,
            key,
        )

    def _get_photo_thumbnail_mds_key(self, type_, photo_id):
        return self._get_photo_mds_key(type_, photo_id) + '_thumbnail'

    def _get_photo_thumbnail_url(self, key):
        return self._get_photo_url(key) + '_thumbnail'
