import concurrent.futures
import enum
import logging

import furl

from kubiki.util import make_requests_session

import cars.settings


LOGGER = logging.getLogger(__name__)


class TelematicsProxy(object):

    # end_lease() reports success before window closers start working.
    # They work for some time and while they're active no other command can proceed.
    POST_END_LEASE_DELAY = 7

    def __init__(self, backends, retries, timeout):
        self._backend_furls = [furl.furl(backend) for backend in backends]
        self._timeout = timeout

        pool_size = 3 * len(self._backend_furls)
        self._pool = concurrent.futures.ThreadPoolExecutor(max_workers=pool_size)
        self._session = make_requests_session(retries=retries, pool_maxsize=pool_size)

    @classmethod
    def from_settings(cls):
        api_settings = cars.settings.TELEMATICS['api']
        return cls(
            backends=api_settings['backends'],
            retries=api_settings['retries'],
            timeout=api_settings['timeout'],
        )

    def blink(self, imei):
        return self._send_action(imei, 'blink')

    def close(self, imei):
        return self._send_action(imei, 'closealldoors')

    def open(self, imei):
        return self._send_action(imei, 'openalldoors')

    def start_engine(self, imei):
        return self._send_action(imei, 'startengine')

    def stop_engine(self, imei):
        return self._send_action(imei, 'stopengine')

    def start_lease(self, imei):
        return self._send_action(imei, 'startlease')

    def end_lease(self, imei):
        return self._send_action(imei, 'endlease')

    def warmup(self, imei):
        return self._send_action(imei, 'warming')

    def stop_warmup(self, imei):
        return self._send_action(imei, 'stopwarming')

    def lock_hood(self, imei):
        return self._send_action(imei, 'lockhood')

    def unlock_hood(self, imei):
        return self._send_action(imei, 'unlockhood')

    def _send_action(self, imei, action):
        LOGGER.info('telematics action %s for %s', action, imei)

        backend_urls = self._make_urls(imei, action)
        response = self._make_requests(urls=backend_urls)

        if response.status is TelematicsApiResponse.Status.NOT_CONNECTED:
            LOGGER.warning('action %s failed: %s not connected', action, imei)

        return response

    def _make_urls(self, imei, action):
        path = '/api/cars/{}/command/{}/'.format(imei, action)
        action_urls = [b.copy().set(path=path) for b in self._backend_furls]
        return action_urls

    def _make_requests(self, urls, data=None):
        futures = []
        for url in urls:
            future = self._pool.submit(
                self._session.post,
                url=url,
                json=data,
                timeout=self._timeout,
            )
            futures.append(future)

        found = False
        for future in concurrent.futures.as_completed(futures):
            try:
                response = future.result()
                if response.status_code != 404:
                    response.raise_for_status()
                if response.ok:
                    found = True
                    break
            except Exception:
                LOGGER.exception('telematics request failed')

        if not found:
            return TelematicsApiResponse.not_connected()

        return TelematicsApiResponse.from_response(response)


class TelematicsApiResponse(object):

    class Error(Exception):

        def __init__(self, *args, code, **kwargs):
            super().__init__(*args, **kwargs)
            self.code = code

        def __repr__(self):
            return self.code

    class Status(enum.Enum):
        OK = 'ok'
        ERROR = 'error'
        NOT_CONNECTED = 'not_connected'
        TIMEOUT = 'timeout'

    def __init__(self, status, code):
        self.status = status
        self.code = code

    def __repr__(self):
        return '<TelematicsApiResponse: status={}, code={}>'.format(self.status.name, self.code)

    @classmethod
    def from_response(cls, response):
        data = response.json()
        status = cls.Status(data['status'])
        code = data['code']
        return cls(
            status=status,
            code=code,
        )

    @classmethod
    def not_connected(cls):
        return cls(
            status=cls.Status.NOT_CONNECTED,
            code='not_connected',
        )

    def raise_for_status(self):
        if self.status is not self.Status.OK:
            code = self.code or 'unknown'
            raise self.Error(code=code)


class TelematicsProxyStub(object):

    POST_END_LEASE_DELAY = 0

    @classmethod
    def from_settings(cls):
        return cls()

    def _make_ok_response(self):
        return TelematicsApiResponse(
            status=TelematicsApiResponse.Status.OK,
            code=None,
        )

    def blink(self, imei):
        LOGGER.info('TelematicsProxyStub.blink(%s)', imei)
        return self._make_ok_response()

    def close(self, imei):
        LOGGER.info('TelematicsProxyStub.close(%s)', imei)
        return self._make_ok_response()

    def open(self, imei):
        LOGGER.info('TelematicsProxyStub.open(%s)', imei)
        return self._make_ok_response()

    def start_engine(self, imei):
        LOGGER.info('TelematicsProxyStub.start_engine(%s)', imei)
        return self._make_ok_response()

    def stop_engine(self, imei):
        LOGGER.info('TelematicsProxyStub.stop_engine(%s)', imei)
        return self._make_ok_response()

    def start_lease(self, imei):
        LOGGER.info('TelematicsProxyStub.start_lease(%s)', imei)
        return self._make_ok_response()

    def end_lease(self, imei):
        LOGGER.info('TelematicsProxyStub.end_lease(%s)', imei)
        return self._make_ok_response()

    def warmup(self, imei):
        LOGGER.info('TelematicsProxyStub.warmup(%s)', imei)
        return self._make_ok_response()

    def stop_warmup(self, imei):
        LOGGER.info('TelematicsProxyStub.stop_warmup(%s)', imei)
        return self._make_ok_response()

    def lock_hood(self, imei):
        LOGGER.info('TelematicsProxyStub.lock_hood(%s)', imei)
        return self._make_ok_response()

    def unlock_hood(self, imei):
        LOGGER.info('TelematicsProxyStub.unlock_hood(%s)', imei)
        return self._make_ok_response()


class OpenDriverDoorTelematicsProxyStub(TelematicsProxyStub):

    def start_lease(self, _):
        return self._make_error_response()

    def end_lease(self, _):
        return self._make_error_response()

    def warmup(self, _):
        return self._make_error_response()

    def _make_error_response(self):
        return TelematicsApiResponse(
            status=TelematicsApiResponse.Status.ERROR,
            code='driver_door_open',
        )
