import datetime
import logging
import time
from math import radians, cos, sin, asin, sqrt

import pytz
from django.utils import timezone

from ..models.car import Car
from ..models.car_location import CarLocation
from ..models.car_telematics_state import CarTelematicsState
from .solomon import SOLOMON
from .wialon_parameters import WialonParameters


LOGGER = logging.getLogger(__name__)


class TelematicsStateUpdater(object):

    class CarDoesNotExistError(Exception):
        pass

    _parameter_by_code = {
        parameter.value.code: parameter.value
        for parameter in WialonParameters
    }

    def __init__(self, imei):
        self._imei = imei

    def update_from_raw_packet(self, packet):
        self._update_telematics_state(packet)
        self._update_position_data(packet)
        self._monitor_lag(packet)

    def _update_telematics_state(self, packet):
        update_payload = {}

        for record in packet['records']:
            record_date = self._parse_record_date(record)

            for subrecord in record['subrecords']:
                if subrecord.get('type') != 'custom_parameters':
                    continue

                params = subrecord['params']
                for code, value in params.items():
                    code = int(code)

                    parameter = self._parameter_by_code.get(code)
                    if parameter is None:
                        continue
                    if value == 0 and not parameter.allow_zeros:
                        LOGGER.info('[%s] skipping zero value for %s', self._imei, parameter)
                        continue

                    update_payload[parameter.db_column] = value
                    update_payload[parameter.updated_at_db_column] = record_date

        if update_payload:
            n_updated = self._do_update_telematics_state(update_payload)
            if n_updated == 0:
                self._create_telematics_state(imei=self._imei)
                n_updated = self._do_update_telematics_state(update_payload)

            if n_updated > 1:
                LOGGER.error(
                    'updated more that one telematics state for imei %s: %s',
                    self._imei,
                    n_updated,
                )
            else:
                LOGGER.info('successfully updated %s with %s', self._imei, packet)

    def _do_update_telematics_state(self, payload):
        return (
            CarTelematicsState.objects
            .filter(car__imei=self._imei)
            .update(**payload)
        )

    def _create_telematics_state(self, imei):
        try:
            car = Car.objects.get(imei=imei)
        except Car.DoesNotExist:
            raise self.CarDoesNotExistError(imei)
        CarTelematicsState.objects.create(car=car)

    def _update_position_data(self, packet):
        latest_position_data_date = None
        latest_position_data = None

        for record in packet['records']:
            record_date = self._parse_record_date(record)

            for subrecord in record['subrecords']:
                if subrecord.get('type') != 'position_data':
                    continue

                if latest_position_data_date is None or record_date > latest_position_data_date:
                    latest_position_data_date = record_date
                    latest_position_data = subrecord

        if latest_position_data is None or latest_position_data_date + datetime.timedelta(seconds=30) < timezone.now():
            return

        if latest_position_data_date > timezone.now() + datetime.timedelta(minutes=10):
            return

        update_kwargs = {
            'updated_at': latest_position_data_date,
            'lat': latest_position_data['lat'],
            'lon': latest_position_data['lon'],
            'course': latest_position_data['course'],
        }

        location = (
            CarLocation.objects
            .filter(car__imei=self._imei)
            .first()
        )
        if location is None:
            self._create_car_location(imei=self._imei, initial_kwargs=update_kwargs)
        elif location.updated_at + datetime.timedelta(seconds=10) >= latest_position_data_date:
            # location is too recent, no need to update
            return
        else:
            if (self._distance(
                    latest_position_data['lat'],
                    latest_position_data['lon'],
                    location.lat,
                    location.lon) > 100.0):
                car = Car.objects.get(imei=self._imei)
                car.update_timestamp = time.time()
                car.updated_at = timezone.now()
                car.save()

            update_qs = (
                CarLocation.objects
                .filter(
                    id=location.id,
                    updated_at__lt=latest_position_data_date,
                )
            )
            if update_kwargs['lat'] == 0 or update_kwargs['lon'] == 0:
                LOGGER.info('missing coordinates for %s: %s', self._imei, update_kwargs)
                update_qs.update(updated_at=latest_position_data_date)
            else:
                update_qs.update(**update_kwargs)

    def _distance(self, lat1, lon1, lat2, lon2):
        lon1, lat1, lon2, lat2 = map(radians, [lon1, lat1, lon2, lat2])
        dlon = lon2 - lon1
        dlat = lat2 - lat1
        a = sin(dlat / 2) ** 2 + cos(lat1) * cos(lat2) * sin(dlon / 2) ** 2
        c = 2 * asin(sqrt(a))
        km = 6371000.0 * c
        return km

    def _create_car_location(self, imei, initial_kwargs):
        try:
            car = Car.objects.get(imei=self._imei)
        except Car.DoesNotExist:
            raise self.CarDoesNotExistError(imei)
        CarLocation.objects.create(car=car, **initial_kwargs)

    def _monitor_lag(self, packet):
        for record in packet['records']:
            record_date = self._parse_record_date(record)
            lag = timezone.now() - record_date
            SOLOMON.set_value(
                'telematics_state_updater.lag',
                lag.total_seconds(),
                labels={
                    'imei': str(self._imei),
                },
            )

    def _parse_record_date(self, record):
        return datetime.datetime.fromtimestamp(record['timestamp'], tz=pytz.UTC)
