import enum
import uuid

from django.db import models

from .car import Car


class CarsharingAcceptance(models.Model):

    class CarCondition(enum.Enum):
        OK = 'ok'
        OK_WITH_ISSUES = 'ok_with_issues'
        UNSATISFACTORY = 'unsatisfactory'

    car = models.ForeignKey(
        Car,
        on_delete=models.CASCADE,
        related_name='acceptances',
    )

    car_condition = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in CarCondition],
        null=True,
    )
    fuel_card_present = models.NullBooleanField()
    insurance_present = models.NullBooleanField()
    sts_present = models.NullBooleanField()
    comment = models.TextField(null=True)

    class Meta:
        db_table = 'carsharing_acceptance'

    def __str__(self):
        return '<CarsharingAcceptance: car.id={}>'.format(self.car_id)

    def is_completed(self):
        """
        Fuel card check is commented in order not to break working versions, where fuel card
        state is currently not supported.
        Uncomment when EKA stuff is fully in production.

        if self.fuel_card_present is None:
            return False
        """
        if self.insurance_present is None:
            return False
        if not self.sts_present:
            return False
        if self.car_condition is None:
            return False

        return True


class CarsharingAcceptancePhoto(models.Model):

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)

    acceptance = models.ForeignKey(
        CarsharingAcceptance,
        related_name='photos',
        on_delete=models.CASCADE,
    )

    link = models.CharField(max_length=256)

    mime_type = models.CharField(max_length=16)

    submitted_at = models.DateTimeField()

    class Meta:
        db_table = 'carsharing_acceptance_photo'
