import enum
import time
import uuid

from django.core.exceptions import ObjectDoesNotExist
from django.db import models

from cars.carsharing.models.car_hardware import CarHardwareBeacon, CarHardwareHead, CarHardwareModem, CarHardwareVega
from .car_model import CarModel


class Car(models.Model):

    class Status(enum.Enum):
        ACCEPTANCE = 'acceptance'
        AVAILABLE = 'available'
        CLEANING = 'cleaning'
        FUELING = 'fueling'
        NEW = 'new'
        PARKING = 'parking'
        RESERVATION = 'reservation'
        RESERVATION_PAID = 'reservation_paid'
        RIDE = 'ride'
        SERVICE = 'service'

        OLD_STATE_ACCEPTANCE = 'old_state_accept'
        OLD_STATE_RESERV = 'old_state_reserv'
        OLD_STATE_RIDING = 'old_state_riding'
        OLD_STATE_PARKING = 'old_state_parkin'

    id = models.UUIDField(default=uuid.uuid4, primary_key=True)
    vin = models.CharField(max_length=17, unique=True, null=True)
    contract_id = models.CharField(max_length=32, unique=True, null=True)
    registration_id = models.BigIntegerField(null=True, unique=True)
    registration_date = models.DateTimeField(null=True)
    imei = models.BigIntegerField(null=True, unique=True)
    number = models.CharField(max_length=16, unique=True, null=True)
    model = models.ForeignKey(
        CarModel,
        db_column='model_code',
        on_delete=models.CASCADE,
    )
    status = models.CharField(
        max_length=16,
        choices=[(x.value, x.name) for x in Status],
        default=Status.NEW.value,
    )
    fuel_card_number = models.CharField(max_length=18, null=True)
    responsible_picker = models.CharField(max_length=128, null=True)
    update_timestamp = models.IntegerField(default=time.time, null=True)

    beacon = models.ForeignKey(
        CarHardwareBeacon,
        null=True,
        db_column='beacon',
        on_delete=models.CASCADE,
    )
    head = models.ForeignKey(
        CarHardwareHead,
        null=True,
        db_column='head',
        on_delete=models.CASCADE,
    )
    modem = models.ForeignKey(
        CarHardwareModem,
        null=True,
        db_column='modem',
        on_delete=models.CASCADE,
    )
    vega = models.ForeignKey(
        CarHardwareVega,
        null=True,
        db_column='vega',
        on_delete=models.CASCADE,
    )

    # This value determines when the client
    # will receive an updated car to update it on the map.
    updated_at = models.DateTimeField(null=True)

    class Meta:
        db_table = 'car'
        indexes = [
            models.Index(fields=['status'], name='car_status_idx'),
            models.Index(fields=['updated_at'], name='car_updated_at_idx'),
            models.Index(
                fields=['update_timestamp'],
                name='car_update_timestamp_idx',
            ),
        ]

    def __str__(self):
        return str(self.number)

    def get_status(self):
        return self.Status(self.status)

    def get_location(self):
        try:
            location = self.location
        except ObjectDoesNotExist:
            location = None
        return location

    def get_telematics_state(self):
        try:
            telematics_state = self.telematics_state
        except ObjectDoesNotExist:
            telematics_state = None
        return telematics_state

    def get_mileage(self):
        telematics_state = self.get_telematics_state()
        if telematics_state is None:
            return None
        else:
            return telematics_state.mileage

    def get_insurance(self):
        try:
            insurance = self.insurance
        except ObjectDoesNotExist:
            insurance = None
        return insurance
