import enum
import json
import uuid

from django.db import models
from django.utils import timezone

from .car_insurance import CarInsurance
from .car_hardware import (
    CarHardwareBeacon,
    CarHardwareHead,
    CarHardwareModem,
    CarHardwareSim,
    CarHardwareVega,
)
from .car_registry_document import CarRegistryDocument
from cars.users.models import User
from .car import Car


class CarDocumentManager(models.Manager):

    def with_related(self):
        return (
            self.get_queryset()
            .select_related(
                'car_hardware_beacon__sim',
                'car_hardware_head',
                'car_hardware_modem__sim',
                'car_hardware_sim',
                'car_hardware_vega__primary_sim',
                'car_hardware_vega__secondary_sim',
            )
        )


class CarDocument(models.Model):

    class Type(enum.Enum):
        CAR_HARDWARE_BEACON = 'car_hardware_beacon'
        CAR_HARDWARE_HEAD = 'car_hardware_head'
        CAR_HARDWARE_MODEM = 'car_hardware_modem'
        CAR_HARDWARE_SIM = 'car_hardware_sim'
        CAR_HARDWARE_VEGA = 'car_hardware_vega'
        CAR_INSURANCE_POLICY = 'car_insurance_policy'
        CAR_REGISTRY_DOCUMENT = 'car_registry_document'

    id = models.UUIDField(primary_key=True, default=uuid.uuid4)

    type = models.CharField(max_length=32, choices=[(x.value, x.name) for x in Type])

    added_at = models.DateTimeField()
    modified_at = models.DateTimeField(default=timezone.now)

    added_by = models.ForeignKey(
        User,
        related_name='car_documents_added',
        on_delete=models.CASCADE,
        db_index=False,
        null=True,
    )

    car_hardware_beacon = models.OneToOneField(
        CarHardwareBeacon,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_hardware_head = models.OneToOneField(
        CarHardwareHead,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_hardware_modem = models.OneToOneField(
        CarHardwareModem,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_hardware_sim = models.OneToOneField(
        CarHardwareSim,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_hardware_vega = models.OneToOneField(
        CarHardwareVega,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_insurance_policy = models.OneToOneField(
        CarInsurance,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    car_registry_document = models.OneToOneField(
        CarRegistryDocument,
        related_name='car_document',
        on_delete=models.CASCADE,
        null=True,
    )

    blob = models.CharField(max_length=8196, null=True)

    objects = CarDocumentManager()

    class Meta:
        db_table = 'car_document'
        """
        db_constraints = {
            'exclusive_arc_chk': (
                '''
                CHECK (
                  (
                    (type = 'car_hardware_beacon' AND car_hardware_beacon_id IS NOT NULL)
                    OR
                    (type = 'car_hardware_head' AND car_hardware_head_id IS NOT NULL)
                    OR
                    (type = 'car_hardware_modem' AND car_hardware_modem_id IS NOT NULL)
                    OR
                    (type = 'car_hardware_sim' AND car_hardware_sim_id IS NOT NULL)
                    OR
                    (type = 'car_hardware_vega' AND car_hardware_vega_id IS NOT NULL)
                    OR
                    (type = 'car_insurance_policy' AND car_insurance_policy_id IS NOT NULL)                    
                    OR
                    (type = 'car_registry_document' AND car_registry_document_id IS NOT NULL)
                  )
                  AND
                  (
                    CASE WHEN car_hardware_beacon_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_hardware_head_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_hardware_modem_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_hardware_sim_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_hardware_vega_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_insurance_policy_id IS NOT NULL THEN 1 ELSE 0 END
                    +
                    CASE WHEN car_registry_document_id IS NOT NULL THEN 1 ELSE 0 END
                  ) = 1
                )
                '''
            ),
        }
        """
        indexes = [
            models.Index(
                fields=['added_at'],
                name='added_at_idx',
            ),
            models.Index(
                fields=['modified_at'],
                name='modified_at_idx',
            ),
        ]

    def get_type(self):
        return self.Type(self.type)

    def get_impl(self):
        if self.type in ('car_transponder', 'car_transponder_spb', 'car_airport_pass'):
            return None
        type_ = self.get_type()
        if type_ is self.Type.CAR_HARDWARE_BEACON:
            impl = self.car_hardware_beacon
        elif type_ is self.Type.CAR_HARDWARE_HEAD:
            impl = self.car_hardware_head
        elif type_ is self.Type.CAR_HARDWARE_MODEM:
            impl = self.car_hardware_modem
        elif type_ is self.Type.CAR_HARDWARE_SIM:
            impl = self.car_hardware_sim
        elif type_ is self.Type.CAR_HARDWARE_VEGA:
            impl = self.car_hardware_vega
        elif type_ is self.Type.CAR_INSURANCE_POLICY:
            impl = self.car_insurance_policy
        elif type_ is self.Type.CAR_REGISTRY_DOCUMENT:
            impl = self.car_registry_document
        else:
            impl = None
        return impl


class CarDocumentAssignmentManager(models.Manager):

    def with_related(self):
        return (
            self.get_queryset()
            .select_related(
                'document__car_hardware_beacon__sim',
                'document__car_hardware_head',
                'document__car_hardware_modem__sim',
                'document__car_hardware_sim',
                'document__car_hardware_vega__primary_sim',
                'document__car_hardware_vega__secondary_sim',
                'document__car_insurance_policy',
                'document__car_registry_document',
            )
        )


class CarDocumentAssignment(models.Model):

    id = models.UUIDField(primary_key=True, default=uuid.uuid4)

    document = models.ForeignKey(
        CarDocument,
        related_name='assignments',
        on_delete=models.CASCADE,
    )
    car = models.ForeignKey(
        Car,
        related_name='document_assignments',
        on_delete=models.CASCADE,
    )

    assigned_at = models.DateTimeField()
    assigned_by = models.ForeignKey(
        User,
        related_name='document_assignments',
        on_delete=models.CASCADE,
        null=True,
    )

    unassigned_at = models.DateTimeField(null=True)
    unassigned_by = models.ForeignKey(
        User,
        related_name='document_unassignments',
        on_delete=models.CASCADE,
        null=True,
    )

    objects = CarDocumentAssignmentManager()

    class Meta:
        db_table = 'car_document_assignment'
        indexes = [
            models.Index(
                fields=['assigned_at'],
                name='assigned_at_idx',
            ),
            models.Index(
                fields=['unassigned_at'],
                name='unassigned_at_idx',
            ),
        ]
