import concurrent.futures
import json
import logging
import os
import time

import shapely.wkt
from haversine import haversine
from kazoo.client import KazooClient
from kazoo.exceptions import NodeExistsError, NoNodeError
from kazoo.protocol.states import EventType
from kazoo.recipe.watchers import ChildrenWatch, DataWatch

import cars.settings
from cars.core.filter import CarFilter


LOGGER = logging.getLogger(__name__)


class NoAlertError(Exception):
    pass


class Alert(object):

    def __init__(self, platform, device_id, location, area, filter_,
                 created_at=None, updated_at=None, reminded_at=None):

        self.platform = platform
        self.device_id = device_id
        self.location = location
        self.area = area
        self.filter = filter_

        now = time.time()
        if created_at is None:
            created_at = now
        if updated_at is None:
            updated_at = now

        self.created_at = created_at
        self.updated_at = updated_at
        self.reminded_at = reminded_at  # Last time a user was reminded of the alert.

    def __eq__(self, other):
        return (
            self.platform == other.platform
            and self.device_id == other.device_id
            and self.location.equals(other.location)
            and self.filter == other.filter
        )

    def __str__(self):
        return self.serialize().decode('utf8')

    @property
    def id(self):
        return (self.platform, self.device_id)

    @property
    def location_lat_lon(self):
        return (self.location.coords[0][1], self.location.coords[0][0])

    def update(self, prev_alert):
        '''Update properties of this alert given that it's the successor of prev_alert'''

        assert self.platform == prev_alert.platform
        assert self.device_id == prev_alert.device_id

        self.created_at = prev_alert.created_at
        if self.reminded_at is None:
            self.reminded_at = prev_alert.reminded_at

        distance = haversine(self.location_lat_lon, prev_alert.location_lat_lon)
        if distance < cars.settings.RADAR['location_debounce_distance']:
            self.location = prev_alert.location
            self.area = prev_alert.area

    def to_dict(self):
        data = {
            'platform': self.platform,
            'device_id': self.device_id,
            'location': self.location.wkt,
            'area': self.area.wkt,
            'filter': self.filter.to_dict(),
            'created_at': self.created_at,
            'updated_at': self.updated_at,
            'reminded_at': self.reminded_at,
        }
        return data

    def serialize(self):
        data = self.to_dict()
        serialized_data = json.dumps(data).encode('utf8')
        return serialized_data

    @classmethod
    def deserialize(cls, data):
        data = data.decode('utf8')
        data = json.loads(data)
        return cls(platform=data['platform'],
                   device_id=data['device_id'],
                   location=shapely.wkt.loads(data['location']),
                   area=shapely.wkt.loads(data['area']),
                   filter_=CarFilter.from_dict(data['filter']),
                   created_at=data['created_at'],
                   updated_at=data['updated_at'],
                   reminded_at=data['reminded_at'])


def get_alerts_backend():
    return ZookeeperAlertsBackend(**cars.settings.ZOOKEEPER)


def get_alerts_listener():
    return ZookeeperAlertsListener(**cars.settings.ZOOKEEPER)


class ZookeeperClientManager(object):

    # Persist Zookeeper connections between Django requests.
    _clients = {}  # Connection string -> KazooClient.

    @classmethod
    def get_client(cls, hosts):
        hosts = sorted(hosts)
        conn_str = ','.join(hosts)
        client = cls._clients.get(conn_str)
        if not client:
            client = KazooClient(hosts=conn_str)
            client.start()
            cls._clients[conn_str] = client
        return client


class ZookeeperAlertsPathBuilder(object):

    def __init__(self, root):
        self._root = root

    @property
    def root(self):
        return self._root

    @property
    def index(self):
        return os.path.join(self.root, 'index')

    def get_alert_path(self, platform, device_id):
        return os.path.join(self.index, platform, device_id)

    def get_device_ids(self, platform, client):
        return client.get_children(self.get_platform_dir(platform))

    def get_platform_dir(self, platform):
        return os.path.join(self.index, platform)

    def get_platform_by_dir(self, directory):
        return os.path.basename(directory)

    def get_platforms(self, client):
        return client.get_children(self.index)


class ZookeeperAlertsBackend(object):

    _path_builder = None

    def __init__(self, hosts, workdir):
        self._client = ZookeeperClientManager.get_client(hosts=hosts)
        self.setup(client=self._client, workdir=workdir)

    @classmethod
    def setup(cls, client, workdir):
        if cls._path_builder is None:
            cls._path_builder = ZookeeperAlertsPathBuilder(root=workdir)
            client.ensure_path(cls._path_builder.index)
        else:
            assert workdir == cls._path_builder.root

    def get_alert(self, platform, device_id):
        return self._client.retry(self._do_get_alert, platform, device_id)

    def _do_get_alert(self, platform, device_id):
        path = self._path_builder.get_alert_path(platform=platform, device_id=device_id)

        try:
            serialized_alert, _ = self._client.get(path)
        except NoNodeError:
            serialized_alert = None

        if serialized_alert:
            alert = Alert.deserialize(serialized_alert)
        else:
            alert = None

        return alert

    def cancel_alert(self, platform, device_id):
        return self._client.retry(self._do_cancel_alert, platform, device_id)

    def _do_cancel_alert(self, platform, device_id):
        path = self._path_builder.get_alert_path(platform=platform, device_id=device_id)
        try:
            self._client.delete(path)
        except NoNodeError:
            raise NoAlertError

    def upsert_alert(self, alert):
        return self._client.retry(self._do_upsert_alert, alert)

    def _do_upsert_alert(self, alert):
        prev_alert = self.get_alert(platform=alert.platform, device_id=alert.device_id)
        if prev_alert:
            alert.update(prev_alert)

        path = self._path_builder.get_alert_path(platform=alert.platform,
                                                 device_id=alert.device_id)
        serialized_alert = alert.serialize()

        if self._client.exists(path):
            self._client.set(path, serialized_alert)
        else:
            try:
                # Don't raise an exception if a concurrent modification happens.
                self._client.create(path, serialized_alert, makepath=True)
            except NodeExistsError:
                pass


class ZookeeperAlertsListener(object):

    def __init__(self, hosts, workdir):
        self._client = ZookeeperClientManager.get_client(hosts=hosts)
        self._path_builder = ZookeeperAlertsPathBuilder(root=workdir)

        self._alerts = {}  # (platform, device_id) -> Alert

        self._on_alert_updated_callbacks = []

        self._lock = self._client.Lock(os.path.join(workdir, 'lock'))
        self._lock.acquire()

    @property
    def alerts(self):
        return list(self._alerts.values())

    def setup(self):
        pool = concurrent.futures.ThreadPoolExecutor(max_workers=128)
        futures = []

        platforms = self._path_builder.get_platforms(client=self._client)
        for platform in platforms:
            platform_dir = self._path_builder.get_platform_dir(platform)
            ChildrenWatch(
                client=self._client,
                path=platform_dir,
                func=self._make_on_platform_changed_handler(platform),
                send_event=True,
            )

        for future in futures:
            future.result()

        pool.shutdown()

        LOGGER.info('Alerts listener initialized with %s alerts', len(self._alerts))

    def _make_on_platform_changed_handler(self, platform):
        '''Make ZK watcher to be notified on new and cancelled alerts.'''

        def _on_platform_changed(children, event):
            '''Set watches on new alerts.'''

            if event and event.type != EventType.CHILD:
                LOGGER.warning('Unexpected event: %s', event)
                return

            for device_id in children:
                alert_id = (platform, device_id)
                if alert_id not in self._alerts:
                    alert_path = self._path_builder.get_alert_path(platform=platform,
                                                                   device_id=device_id)
                    DataWatch(
                        client=self._client,
                        path=alert_path,
                        func=self._make_on_alert_changed_handler(platform, device_id),
                    )

        return _on_platform_changed

    def _make_on_alert_changed_handler(self, platform, device_id):

        def on_alert_changed(data, stat, event=None):  # pylint: disable=unused-argument
            alert_id = (platform, device_id)
            old_alert = self._alerts.get(alert_id)

            if data is None:
                if alert_id in self._alerts:
                    self._alerts.pop(alert_id)
                else:
                    LOGGER.error('Alert to be removed was not tracked')
                alert = None
            else:
                alert = Alert.deserialize(data)
                self._alerts[alert.id] = alert

            for callback in self._on_alert_updated_callbacks:
                callback(old_alert, alert)

        return on_alert_changed

    def _load_alert(self, platform, device_id):
        alert_path = self._path_builder.get_alert_path(platform=platform, device_id=device_id)
        data, _ = self._client.get(alert_path)
        alert = Alert.deserialize(data)
        return alert

    def listen(self):
        while True:
            time.sleep(1)

    def add_alert_updated_callback(self, callback):
        self._on_alert_updated_callbacks.append(callback)
