import logging

from django_yauth.authentication_mechanisms.base import BaseMechanism
from django_yauth.user import AnonymousYandexUser, YandexUser

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.users.models.user import User

LOGGER = logging.getLogger(__name__)


class Mechanism(BaseMechanism):
    saas_client = SaasDriveAdminClient.from_settings()

    if 'admin.' in saas_client.urls.host:
        fallback_saas_client = SaasDriveAdminClient.from_settings(
            root_url='https://prestable.carsharing.yandex.net'
        )
    else:
        fallback_saas_client = None

    def extract_params(self, request):
        return {'request': request}

    def apply(self, request):
        data = self.saas_client.get_raw_request_user_actions(request, re_raise=False)

        if data is None and self.fallback_saas_client is not None:
            data = self.fallback_saas_client.get_raw_request_user_actions(request, re_raise=False)

        if data is None:
            LOGGER.error('no user actions data received')
            return self.anonymous()

        if 'user_id' not in data:
            LOGGER.error('no user_id provided in response')
            return self.anonymous()

        try:
            user = User.objects.get(id=data['user_id'])
        except Exception:
            LOGGER.exception('no user with id {} exists'.format(data['user_id']))
            return self.anonymous()

        request._user = user  # pay attention, that 'user' attribute is not set and to be provided by a middleware

        return YandexUser(uid=user.uid, mechanism=self)

    def anonymous(self, session=None):
        """
        Этот метод для совместимости с yauth.
        Если вы используете только django AUTHENTICATION_BACKENDS
        вам достаточно возвращать None вместо AnonymousUser.
        Бекэнды yauth про это знают и будут отрабатывать как нужно.
        """
        return AnonymousYandexUser(blackbox_result=session, mechanism=self)
