import itertools

from rest_framework.permissions import BasePermission
from rest_framework.exceptions import AuthenticationFailed, PermissionDenied

from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView

from cars.core.saas_drive_admin import SaasDriveAdminClient
from cars.users.models import UserRole


def _prepare_collection(collection):
    if collection is None:
        prepared_collection = ()
    elif not isinstance(collection, (tuple, list)):
        prepared_collection = (collection, )
    else:
        prepared_collection = collection
    return prepared_collection


class DriveRolePermissionFactory(object):
    @classmethod
    def build(cls, required_user_roles, *, require_all=False, use_proxy=True):
        required_user_roles = _prepare_collection(required_user_roles)

        class DriveRolePermission(BasePermission):
            saas_client = SaasDriveAdminClient.from_settings()

            if 'admin.' in saas_client.urls.host:
                fallback_saas_client = SaasDriveAdminClient.from_settings(
                    root_url='https://prestable.carsharing.yandex.net'
                )
            else:
                fallback_saas_client = None

            _required_user_roles = required_user_roles
            _require_all = require_all
            _use_proxy = use_proxy

            @property
            def message(self):
                return 'user has no required roles ({})'.format(self._required_user_roles)

            def has_permission(self, request, view):
                has_permission = self.saas_client.check_user_role(
                    *self._required_user_roles, request=request, require_all=require_all, use_proxy=self._use_proxy
                )

                retry_check = (
                    not has_permission and
                    self.fallback_saas_client is not None and
                    not self.saas_client.get_cached_request_user_roles(request, [])
                )

                if retry_check:
                    has_permission = self.fallback_saas_client.check_user_role(
                        *self._required_user_roles, request=request, require_all=require_all, use_proxy=self._use_proxy
                    )

                return has_permission

        return DriveRolePermission


class DriveActionPermissionFactory(object):
    @classmethod
    def build(cls, required_user_actions, *, require_all=False, use_proxy=True):
        required_user_actions = _prepare_collection(required_user_actions)

        class DriveActionPermission(BasePermission):
            saas_client = SaasDriveAdminClient.from_settings()

            if 'admin.' in saas_client.urls.host:
                fallback_saas_client = SaasDriveAdminClient.from_settings(
                    root_url='https://prestable.carsharing.yandex.net'
                )
            else:
                fallback_saas_client = None

            _required_user_actions = required_user_actions
            _require_all = require_all
            _use_proxy = use_proxy

            @property
            def message(self):
                return 'user has no required actions ({})'.format(self._required_user_actions)

            def has_permission(self, request, view):
                has_permission = self.saas_client.check_user_action(
                    *self._required_user_actions, request=request, require_all=require_all, use_proxy=self._use_proxy
                )

                retry_check = (
                    not has_permission and
                    self.fallback_saas_client is not None and
                    not self.saas_client.get_cached_request_user_actions(request, [])
                )

                if retry_check:
                    has_permission = self.fallback_saas_client.check_user_action(
                        *self._required_user_actions, request=request, require_all=require_all, use_proxy=self._use_proxy
                    )

                return has_permission

        return DriveActionPermission


class DrivePermissionAPIView(CarsharingAPIView):
    permission_classes = [IsAuthenticated]
    role_permission_classes = []  # List[DriveRolePermissionFactory.build(NotImplemented)]
    action_permission_classes = []  # List[DriveActionPermissionFactory.build(NotImplemented)]

    def check_banned_roles(self, user):
        roles = (
            UserRole.objects
            .filter(
                user=user,
                role__role_id__in=[
                    'team_adm_level1',
                    'team_adm_level2',
                    'team_adm_level3'
                ],
                active=True,
            )
        )
        return roles.count() == 0

    def get_permissions(self):
        return [
            permission() for permission in itertools.chain(
                self.permission_classes, self.role_permission_classes, self.action_permission_classes
            )
        ]
