import hashlib

from rest_framework.exceptions import AuthenticationFailed
from rest_framework.authentication import BasicAuthentication

from cars.django.permissions import IsAuthenticated
from cars.django.views import CarsharingAPIView


class ExternalUser(object):
    def __init__(self, username):
        self._username = username

    @property
    def id(self):
        return 0

    @property
    def uid(self):
        return 0

    @property
    def is_anonymous(self):
        return False

    @property
    def is_authenticated(self):
        return True

    def get_username(self):
        return self._username


class ExternalAuthHelper(object):
    MAX_USER_NAME_LENGTH = 255

    def __init__(self, token, salt=''):
        self._token = token
        self._salt = salt

    def get_expected_token_hash(self, username):
        data = username + self._token + self._salt
        return hashlib.sha256(data.encode('ascii')).hexdigest()

    def authorize_request(self, username, token_hash, validate_token=False):
        if not (username and token_hash and len(username) <= self.MAX_USER_NAME_LENGTH):
            raise AuthenticationFailed('invalid username/password')

        if validate_token:
            is_token_valid = self._validate_token()
            if not is_token_valid:
                raise AuthenticationFailed('invalid token')

        expected_token_hash = self.get_expected_token_hash(username)
        is_authorized = (expected_token_hash == token_hash)

        if not is_authorized:
            raise AuthenticationFailed('invalid username/password')

        return ExternalUser(username), None

    def _validate_token(self):
        # as token is stored locally, validation is not required
        return True


class ExternalAuthenticationFactory(object):
    @classmethod
    def build(cls, token, salt=''):

        class ExternalAuthentication(BasicAuthentication):
            auth_helper = ExternalAuthHelper(token=token, salt=salt)

            def authenticate_credentials(self, userid, password, request=None):
                return self.auth_helper.authorize_request(userid, password)

        return ExternalAuthentication


class ExternalAuthenticationAPIView(CarsharingAPIView):
    authentication_classes = [ExternalAuthenticationFactory.build(NotImplemented)]
    permission_classes = [IsAuthenticated]

    def perform_authentication(self, request):
        request.auth
