import base64
import hashlib
import hmac
import logging
import time
import random

import furl
from requests.exceptions import RequestException

import yt.wrapper as yt
from kubiki.util import make_requests_session

from cars.core.util import make_yt_client
import cars.settings


LOGGER = logging.getLogger(__name__)


class AutoCodeAuthentication:

    def __init__(self, system_id, secret_id):
        self._system_id = system_id
        self._secret_id = secret_id

    @classmethod
    def from_settings(cls):
        settings = cars.settings.THIRD_PARTY_APIS['autocode']
        return cls(
            system_id=settings['system_id'],
            secret_id=settings['secret_id'],
        )

    def get_auth_data(self):
        cur_timestamp = int(time.time())
        nonce = self._custom_sha1(str(random.random()))
        sign_str = '{}&{}&{}&{}'.format(
            self._secret_id,
            self._system_id,
            nonce,
            str(cur_timestamp),
        )
        token = self._custom_hash(sign_str, self._secret_id)
        return {
            'Nonce': nonce,
            'SystemId': self._system_id,
            'Timestamp': cur_timestamp,
            'Token': token,
        }

    def _custom_hash(self, message, secret_key):
        hash_bytes = (
            hmac.new(
                secret_key.encode('ascii'),
                message.encode('ascii'),
                hashlib.sha1
            )
            .digest()
        )
        return base64.b64encode(hash_bytes).decode('ascii')

    def _custom_sha1(self, str_to_hash):
        raw_sha1 = hashlib.sha1(str_to_hash.encode('ascii'))
        return raw_sha1.hexdigest().upper()


class AutoCodeBaseClient:

    def __init__(self, auth, yt_client=None, log_table_path=None, request_timeout=300, retries=3):
        assert log_table_path is None or yt_client is not None
        self._auth = auth
        self._request_timeout = request_timeout
        self._yt = yt_client
        self._log_table_path = log_table_path
        self._retries = retries

    @classmethod
    def from_settings(cls):
        return cls(
            auth=AutoCodeAuthentication.from_settings(),
        )

    def _perform_request(self, url, payload):
        _session = make_requests_session(retries=self._retries)

        payload['SecurityInfoData'] = self._auth.get_auth_data()

        try:
            response = _session.post(
                url=url,
                json=payload,
                timeout=self._request_timeout,
            )
            response.raise_for_status()
        except RequestException:
            LOGGER.exception('Exception while querying autocode.')
            raise

        result = response.json()
        payload['SecurityInfoData'] = '<hidden>'

        if self._yt is not None and self._log_table_path is not None:
            try:
                self._yt.write_table(
                    self._log_table_path,
                    [{
                        'timestamp': int(time.time()),
                        'url': url,
                        'request_data': payload,
                        'response_data': result,
                    }]
                )
            except Exception:
                LOGGER.exception('failed to save autocode log to yt')

        return result

    def merge_logtable_chunks(self):
        if self._yt is not None:
            self._yt.run_merge(
                self._log_table_path,
                self._log_table_path,
                spec={'combine_chunks': True}
            )


class AutoCodeDrivingLicenseVerifier(AutoCodeBaseClient):

    DRIVING_LICENSE_INFO_URL = 'http://avtokodapi.mos.ru:7007/mosru/api/driverlicenses'

    def get_driving_license_data(self, license_number, issue_date):
        payload = {
            'Number': license_number,
            'IssueDate': issue_date.strftime('%d.%m.%Y'),
        }
        return self._perform_request(self.DRIVING_LICENSE_INFO_URL, payload)


class AutoCodeFinesGetter(AutoCodeBaseClient):
    MAX_VEHICLES_BATCH = 10000

    BASE_HOST = furl.furl('http://avtokodapi.mos.ru:7102')
    ROUTES = {
        'subscription': 'api/Subscribers/Add',
        'unsubscription': 'api/Subscribers/Remove',
        'fines': 'api/Penalties/Get',
        'paid_fines': 'api/Penalties/GetPaid',
        'fine_photos': 'api/PenaltyPhotos/Get',
        'object_acquisition_confirmation': 'api/Penalties/ConfirmReceived',
    }

    @classmethod
    def from_settings(cls):
        logs_table = cars.settings.THIRD_PARTY_APIS['autocode']['logs_table']
        return cls(
            auth=AutoCodeAuthentication.from_settings(),
            yt_client=make_yt_client('data'),
            log_table_path=yt.TablePath(
                logs_table,
                append=True
            ) if logs_table else None,
        )

    def subscribe_persons(self, driving_license_numbers):
        return self._batch_action(driving_license_numbers, 'DriverLicense', 'subscribe')

    def subscribe_vehicles(self, sts_numbers):
        limit = self.MAX_VEHICLES_BATCH
        for offset in range(0, len(sts_numbers), limit):
            sts_numbers_slice = sts_numbers[offset:offset + limit]
            self._batch_action(sts_numbers_slice, 'Sts', 'subscription')

    def unsubscribe_persons(self, driving_license_numbers):
        return self._batch_action(driving_license_numbers, 'DriverLicense', 'unsubscribe')

    def unsubscribe_vehicles(self, sts_numbers):
        return self._batch_action(sts_numbers, 'Sts', 'unsubscribe')

    def get_new_fines(self, limit=10, offset=0):
        payload = {
            'limit': limit,
            'offset': offset,
        }
        return self._perform_request(
            url=self._get_url('fines'),
            payload=payload,
        )

    def get_new_paid_fines(self, limit=10, offset=0):
        payload = {
            'limit': limit,
            'offset': offset,
        }
        return self._perform_request(
            url=self._get_url('paid_fines'),
            payload=payload,
        )

    def get_violation_photos(self, ruling_number):
        payload = {
            'rulingNumber': ruling_number,
        }
        return self._perform_request(
            url=self._get_url('fine_photos'),
            payload=payload,
        ).get('photos', [])

    def remove_objects_from_feed(self, ids):
        payload = {
            'idList': ids,
        }
        return self._perform_request(
            url=self._get_url('object_acquisition_confirmation'),
            payload=payload,
        )

    def _batch_action(self, numbers, type_, action):
        payload = {
            'subscribers': [{'DocumentType': type_, 'DocumentNumber': number} for number in numbers],
        }
        return self._perform_request(
            url=self._get_url(action),
            payload=payload,
        )

    def _get_url(self, route):
        return (
            self.BASE_HOST
            .copy()
            .join(self.ROUTES[route])
            .url
        )
