import copy
import logging
import os
import time

if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    import django.db

    import kubiki
    from planex.base import DistributedService, ThreadedWorker

    import cars.settings
    from cars.core.solomon import make_solomon_client

    django.setup()
else:
    class DistributedService:
        pass

    class ThreadedWorker:
        pass


LOGGER = logging.getLogger(__name__)


class CarsharingWorker(ThreadedWorker):
    tick_interval = None  # to be set in sub-class
    re_raise = True

    @property
    def solomon(self):
        return self.service.solomon

    def get_solomon_sensor_prefix(self):
        return self.service.get_solomon_sensor_prefix()

    def tick(self):
        tick_start_ts = time.time()

        try:
            try:
                self._do_tick()
            except (django.db.DatabaseError, django.db.OperationalError):
                # PGaaS PgBouncer may drop connections.
                # Application should retry such errors.
                django.db.close_old_connections()
                self._do_tick()
        except Exception as exc:
            LOGGER.exception('daemon failed')

            if self.re_raise:
                raise

        self.solomon.set_value(
            '{}.tick.duration'.format(self.get_solomon_sensor_prefix()),
            time.time() - tick_start_ts,
        )

    def _do_tick(self):
        return self.service._do_tick()


class CarsharingDaemon(DistributedService):

    tick_interval = '* * * * * *'

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.solomon = self.make_solomon_client()

    def init_workers(self):
        # add multiple workers here; a default one is build
        super().init_workers()

    def construct_default_worker_class(self):
        parent = self

        class DefaultWorker(CarsharingWorker):
            tick_interval = parent.tick_interval

        return DefaultWorker

    def _do_tick(self):
        raise NotImplementedError

    def get_distributed_lock_relative_path(self):
        raise NotImplementedError

    def get_solomon_sensor_prefix(self):
        raise NotImplementedError

    def get_solomon_service(self):
        raise NotImplementedError

    @property
    def logging_config(self):
        cfg = copy.deepcopy(cars.settings.LOGGING)
        service_name = self.__class__.__name__

        handler = {
            'level': 'DEBUG',
            'class': 'logging.handlers.TimedRotatingFileHandler',
            'filename': 'logs/{}.log'.format(service_name),
            'formatter': 'verbose',

            'when': 'midnight',  # rotate at midnight
            'interval': 1,  # (each) midnight
            'backupCount': 7,  # save last week
            'utc': True,  # use UTC time
        }

        cfg['handlers'][service_name] = handler
        cfg['loggers']['']['handlers'].append(service_name)

        return cfg

    def get_configs(self):
        return CarsharingDaemonConfig()

    def get_distributed_lock(self):
        locks_config = copy.deepcopy(cars.settings.YT['locks'])
        base_dir = locks_config.pop('base_dir')
        locks_config['path'] = os.path.join(base_dir, self.get_distributed_lock_relative_path())
        return kubiki.distributed_lock.YtLock(**locks_config)

    def make_solomon_client(self):
        solomon_config = copy.deepcopy(cars.settings.SOLOMON)
        solomon_config['service'] = self.get_solomon_service()
        solomon = make_solomon_client(config=solomon_config)
        return solomon


class CarsharingDaemonConfig(object):
    pass
