import logging
import time

import furl
import requests

from kubiki.util import make_requests_session


LOGGER = logging.getLogger(__name__)


class DataSyncClient:

    class NotFoundError(Exception):
        pass

    def __init__(self, host, endpoint_path, service_ticket, push_client, verify_requests=True):
        self._service_ticket = service_ticket
        self._verify_requests = verify_requests
        self._push_client = push_client

        self._base_furl = furl.furl(host)
        self._base_furl.join(endpoint_path)

        self._session = make_requests_session()

    def get(self, collection, uid, key):
        """
        Get all data for the given uid.
        """
        try:
            return self._http_request(
                method='GET',
                url=self._url_with_collection_and_id(collection, key),
                uid=uid,
                collection=collection,
            )
        except requests.RequestException as e:
            if e.response is not None and e.response.status_code == 404:
                raise self.NotFoundError(collection, uid, key)
            raise

    def delete(self, collection, uid, key):
        """
        Delete entry with the given ID from user's records.
        """
        return self._http_request(
            method='DELETE',
            url=self._url_with_collection_and_id(collection, key),
            uid=uid,
            collection=collection,
        )

    def put(self, collection, uid, new_data):
        """
        Replace an entry with ID=new_data['id'] in user's records.
        """

        return self._http_request(
            method='PUT',
            url=self._url_with_collection_and_id(collection, new_data['id']),
            uid=uid,
            collection=collection,
            json=new_data,
        )

    def post(self, collection, uid, data):
        """
        Post a new entry into user's records.
        """
        return self._http_request(
            method='POST',
            url=self._url_with_collection_and_id(collection, data['id']),
            uid=uid,
            collection=collection,
            json=data,
        )

    def _http_request(self, method, url, uid, collection, **kwargs):
        request_data = {
            'method': method,
            'url': url,
            'timeout': 20,
            'verify': self._verify_requests,
        }
        time_before_request = time.time()

        response = self._session.request(
            method=method,
            url=url,
            timeout=20,
            verify=self._verify_requests,
            headers=self._request_headers(uid),
            **kwargs
        )

        time_elapsed = time.time() - time_before_request
        response_data = {
            'status_code': response.status_code,
            'time_elapsed': time_elapsed,
        }
        self._push_client.log(
            type_='datasync',
            data={
                'collection': collection,
                'request_data': request_data,
                'response_data': response_data,
                'uid': uid,
            },
        )

        response.raise_for_status()

        return response.json()

    def _url_with_collection_and_id(self, collection, key):
        request_furl = self._base_furl.copy()
        request_furl.join('/'.join([collection, key]))

        return request_furl.url

    def _request_headers(self, uid):
        return {
            'X-Ya-Service-Ticket': self._service_ticket.content,
            'X-Uid': str(uid),
        }


class StubDataSyncClient:

    def __init__(self, data=None):
        if data is None:
            data = {}
        self._data = data

    def get(self, collection, uid, key):
        uid = str(uid)
        LOGGER.info('StubDataSyncClient.get(%s, %s, %s)', collection, uid, key)
        return self._data.get(collection, {}).get(uid, {}).get(key)

    def delete(self, collection, uid, key):
        uid = str(uid)
        LOGGER.info('StubDataSyncClient.delete(%s, %s, %s)', collection, uid, key)
        self._data[collection][uid].pop(key)

    def put(self, collection, uid, new_data):
        uid = str(uid)
        key = new_data['id']
        LOGGER.info('StubDataSyncClient.put(%s, %s, %s): %s', collection, uid, key, new_data)
        self._data.setdefault(collection, {}).setdefault(uid, {})[key] = new_data

    def post(self, collection, uid, data):
        uid = str(uid)
        key = data['id']
        LOGGER.info('StubDataSyncClient.post(%s, %s, %s): %s', collection, uid, key, data)
        self._data.setdefault(collection, {}).setdefault(uid, {})[key] = data
