import collections
import enum
import logging
import time

import django.db
from django.db.models import Max
from django.db import transaction
import psycopg2

from .util import datetime_helper


LOGGER = logging.getLogger(__name__)


class HistoryAction(enum.Enum):
    ADD = 'add'
    REMOVE = 'remove'
    UPDATE_DATA = 'update_data'


class HistoryManager(object):
    HistoryAction = HistoryAction

    HistoryFieldNameCollection = collections.namedtuple(
        'HistoryFieldNameCollection', ('event', 'user', 'action', 'timestamp')
    )

    DEFAULT_HISTORY_FIELD_NAMES = {
        'event': 'history_event_id',
        'user': 'history_user_id',
        'action': 'history_action',
        'timestamp': 'history_timestamp',
    }

    WRITE_HISTORY_RETRIES = 3
    WRITE_HISTORY_ATTEMPT_DELAY = 0.001

    db_exceptions = (
        django.db.DatabaseError,
        django.db.OperationalError,
        psycopg2.DatabaseError,
        psycopg2.OperationalError,
    )

    not_set = object()

    def __init__(
            self, history_wrapper_cls, *,
            min_allowed_event_id=None, max_allowed_event_id=None,
            history_field_names=None,
            use_timestamp=True,
            excluded_fields=(),
            coercing_rules=None  # temporary to fix schema incompatibility
    ):
        self._history_wrapper_cls = history_wrapper_cls

        self._history_wrapper_cls_field_names = [
            field.name for field in history_wrapper_cls._meta.get_fields()
            if field.name not in set(excluded_fields) and not field.name.startswith('_')
        ]

        self._min_allowed_event_id = min_allowed_event_id
        self._max_allowed_event_id = max_allowed_event_id

        self._history_field_names = self.HistoryFieldNameCollection(**self.DEFAULT_HISTORY_FIELD_NAMES)
        if history_field_names:
            self._history_field_names = self._history_field_names._replace(**history_field_names)

        self._use_timestamp = use_timestamp

        self._coercing_rules = coercing_rules or {}

    @property
    def history_event_field_name(self):
        return self._history_field_names.event

    @property
    def history_user_field_name(self):
        return self._history_field_names.user

    @property
    def history_action_field_name(self):
        return self._history_field_names.action

    @property
    def history_timestamp_field_name(self):
        return self._history_field_names.timestamp

    def add_entry(self, new_object_state, operator_id, timestamp_override=None):
        self.write_history(
            new_object_state, operator_id, HistoryManager.HistoryAction.ADD, timestamp_override
        )

    def remove_entry(self, new_object_state, operator_id, timestamp_override=None):
        self.write_history(
            new_object_state, operator_id, HistoryManager.HistoryAction.REMOVE, timestamp_override
        )

    def update_entry(self, new_object_state, operator_id, timestamp_override=None):
        self.write_history(
            new_object_state, operator_id, HistoryManager.HistoryAction.UPDATE_DATA, timestamp_override
        )

    def write_history(self, new_object_state, operator_id, action, timestamp_override=None):
        history_record = self._make_history_record(new_object_state, operator_id, action, timestamp_override)

        for _ in range(self.WRITE_HISTORY_RETRIES):
            # History table might be locked with C++ history manager.
            # We should not take the lock here as well (even though we'd hypothetically had to)
            # in order not to make primary backend wait - that can result in degradation.

            # So we basically do an assumption that no locks will take place when we try
            # to write (since they're short, when they happen) and do retries in case we were unlucky.

            try:
                history_record.save()

            except self.db_exceptions:
                django.db.close_old_connections()

                LOGGER.exception('unable to save history record')
                time.sleep(self.WRITE_HISTORY_ATTEMPT_DELAY)

            else:
                break

        else:
            history_timestamp = getattr(history_record, self.history_timestamp_field_name)
            state_to_report = {k: str(v) for k, v in vars(history_record).items() if k.endswith('_id')}
            raise Exception(
                'history has not been saved after multiple attempts: action - {}, timestamp - {}, state - {}'
                .format(action.value, history_timestamp, state_to_report)
            )

    def _make_history_record(self, new_object_state, operator_id, action, timestamp_override):
        assert isinstance(operator_id, str)
        assert isinstance(action, HistoryManager.HistoryAction)

        if timestamp_override is not None:
            history_timestamp = timestamp_override
        else:
            if self._use_timestamp:
                history_timestamp = datetime_helper.timestamp_now(truncate=True)
            else:
                history_timestamp = datetime_helper.utc_now()

        new_object_kwargs = {
            self.history_user_field_name: operator_id,
            self.history_action_field_name: action.value,
            self.history_timestamp_field_name: history_timestamp,
        }

        for field_name in self._history_wrapper_cls_field_names:
            value = getattr(new_object_state, field_name, self.not_set)

            if value is not self.not_set:
                if field_name in self._coercing_rules:
                    value = self._coercing_rules[field_name](value)

                new_object_kwargs[field_name] = value

        history_record = self._history_wrapper_cls(**new_object_kwargs)

        return history_record

    def _get_next_allowed_event_id(self):
        assert self._min_allowed_event_id is not None and self._max_allowed_event_id is not None

        max_history_event_id = (
            self._history_wrapper_cls.objects
            .filter(**{
                self.history_event_field_name + '__gte': self._min_allowed_event_id,
                self.history_event_field_name + '__lt': self._max_allowed_event_id,
            })
            .aggregate(Max(self.history_event_field_name))
            [self.history_event_field_name + '__max']
        ) or self._min_allowed_event_id

        return max_history_event_id + 1
