import math
import random

from PIL import Image, ImageDraw, ImageFont


class Watermarker:

    def __init__(self,
                 base_watermark_size,
                 watermark_visibility,
                 font_size,
                 watermark_first_line,
                 watermark_second_line='',
                 random_padding=True):

        # Save basic parameters
        self._base_watermark_size = base_watermark_size
        self._watermark_visibility = watermark_visibility
        self._font_size = font_size
        self._watermark_first_line = watermark_first_line
        self._watermark_second_line = watermark_second_line
        self._random_padding = random_padding

        # Calculate helper values
        self._full_watermark_size = int(base_watermark_size * math.sqrt(2))
        self._crop_margin = (self._full_watermark_size - base_watermark_size) / 2

    def apply_watermark(self, main_image):
        # Create initial watermark image
        watermark = Image.new('RGBA', (self._full_watermark_size, ) * 2)

        draw = ImageDraw.ImageDraw(watermark, 'RGBA')

        # Font origin: /usr/share/fonts/truetype/freefont/
        font = ImageFont.truetype('FreeMonoBold.ttf', size=self._font_size)

        # Draw the header
        draw.text(
            (15, 10 + self._crop_margin * 3),
            self._watermark_first_line,
            font=font
        )

        draw.text(
            (45, 45 + self._crop_margin * 3),
            self._watermark_second_line,
            font=font
        )

        # Rotate to cover more of the original image, then crop, since we've rotated 45 degrees
        watermark = watermark.rotate(315)
        watermark = watermark.crop((self._crop_margin,
                                    self._crop_margin,
                                    self._full_watermark_size - self._crop_margin,
                                    self._full_watermark_size - self._crop_margin))

        # Create mask and apply watermark
        watermask = watermark.convert('L').point(lambda x: min(x, self._watermark_visibility))
        watermark.putalpha(watermask)

        # Shift watermark a bit in order to avoid a case when it prevents from
        # reading important info
        if self._random_padding:
            offset_x = random.randint(0, 100)
            offset_y = random.randint(0, 100)
        else:
            offset_x = 0
            offset_y = 0

        for c1 in range(0, main_image.size[0], watermark.size[0]):
            for c2 in range(0, main_image.size[1], watermark.size[1]):
                main_image.paste(watermark, (c1 + offset_x, c2 + offset_y), watermask)

        return main_image
