import collections
import io
import logging

import boto3

import cars.settings
from cars.core.mds.push_client import MDSPushClientLogger
from cars.django.lock import UwsgiLock
from .streams import DecryptedStream, EncryptedStream


LOGGER = logging.getLogger(__name__)


mds_push_client = MDSPushClientLogger(
    filename=cars.settings.MDS['logfile'],
    lock=UwsgiLock(),
)


class ObjectDoesNotExist(Exception):
    pass


class MDSClient:

    def __init__(self, access_key_id, secret_access_key,
                 service_name, endpoint_url, verify_ssl, push_client):

        self._session = boto3.session.Session(
            aws_access_key_id=access_key_id,
            aws_secret_access_key=secret_access_key,
        )

        self._client = self._session.client(
            service_name=service_name,
            endpoint_url=endpoint_url,
            verify=verify_ssl,
        )

        self._push_client = push_client
        self._host_url = endpoint_url

    @classmethod
    def from_settings(cls, **extra_settings):
        settings = dict(cars.settings.MDS, **extra_settings)
        return cls(
            access_key_id=settings['access_key_id'],
            secret_access_key=settings['secret_access_key'],
            service_name=settings['service_name'],
            endpoint_url=settings['endpoint_url'],
            verify_ssl=settings['verify_ssl'],
            push_client=mds_push_client,
        )

    def get_object(self, bucket, key):
        try:
            response = self._client.get_object(
                Bucket=bucket,
                Key=key,
            )
        except self._client.exceptions.NoSuchKey:
            raise ObjectDoesNotExist(bucket, key)
        self._push_client.log('get_object', bucket, key, response)
        return response

    def get_object_content(self, bucket, key):
        return self.get_object(bucket=bucket, key=key)['Body'].read()

    def put_object(self, bucket, key, body):
        response = self._client.put_object(
            Bucket=bucket,
            Key=key,
            Body=body,
        )
        self._push_client.log('put_object', bucket, key, response)
        return response

    def upload_fileobj(self, fileobj, bucket, key):
        response = self._client.upload_fileobj(
            Bucket=bucket,
            Key=key,
            Fileobj=fileobj,
        )
        self._push_client.log('upload_fileobj', bucket, key, response)
        return response


class EncryptedMDSClient:

    def __init__(self, client, aes_key):
        self._client = client
        self._aes_key = aes_key

    @classmethod
    def from_settings(cls):
        return cls(
            client=MDSClient.from_settings(),
            aes_key=cars.settings.USERS['documents']['mds']['aes_key'],
        )

    def get_object(self, bucket, key):
        result = self._client.get_object(bucket=bucket, key=key)
        result['Body'] = DecryptedStream(
            stream=result['Body'],
            aes_key=self._aes_key,
            iv=key,
            tag=self._get_tag(bucket, key),
        )
        return result

    def get_object_content(self, bucket, key):
        return self.get_object(bucket=bucket, key=key)['Body'].read()

    def put_object(self, body, bucket, key):
        return self.upload_fileobj(
            fileobj=io.BytesIO(body),
            bucket=bucket,
            key=key,
        )

    def upload_fileobj(self, fileobj, bucket, key):
        encrypted_fileobj = EncryptedStream(
            stream=fileobj,
            aes_key=self._aes_key,
            iv=key,
        )
        response = self._client.upload_fileobj(
            fileobj=encrypted_fileobj,
            bucket=bucket,
            key=key,
        )
        self._put_tag(body=encrypted_fileobj.tag, bucket=bucket, key=key)
        return response

    def _get_tag(self, bucket, key):
        return self._client.get_object_content(bucket, key + '_tag')

    def _put_tag(self, body, bucket, key):
        tag_key = '{}_tag'.format(key)
        result = self._client.put_object(
            bucket=bucket,
            key=tag_key,
            body=body,
        )
        return result


class StubMDSClient:

    storage = collections.defaultdict(dict)

    @classmethod
    def from_settings(cls):
        return cls()

    def get_object(self, bucket, key):
        LOGGER.info('StubMDSClient.get_object %s:%s', bucket, key)
        content = self.storage[bucket][key]
        response = {
            'Body': io.BytesIO(content),
        }
        return response

    def get_object_content(self, bucket, key):
        return self.get_object(bucket=bucket, key=key)['Body'].read()

    def put_object(self, body, bucket, key):
        LOGGER.info('StubMDSClient.put_object %s:%s', bucket, key)
        self.storage[bucket][key] = body

    def upload_fileobj(self, fileobj, bucket, key):
        return self.put_object(body=fileobj.read(), bucket=bucket, key=key)
