import logging
import threading
import queue

from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import (
    Cipher, algorithms, modes
)


LOGGER = logging.getLogger(__name__)


class DecryptedStream:

    def __init__(self, stream, aes_key, iv, tag):
        self._stream = stream
        LOGGER.debug('initializing decryptor')
        self._decryptor = Cipher(
            algorithms.AES(aes_key),
            modes.GCM(iv.encode('utf-8'), tag),
            backend=default_backend(),
        ).decryptor()
        self._decryptor.authenticate_additional_data(b'')

        self._is_download_finished = threading.Event()
        self._decrypted_pieces = queue.Queue(maxsize=1)

    def read(self, *args, **kwargs):
        data = self._stream.read(*args, **kwargs)
        if data:
            decrypted_data = self._decryptor.update(data)
        else:
            LOGGER.debug('finalizing decryptor')
            decrypted_data = self._decryptor.finalize()
        return decrypted_data


class EncryptedStream:

    def __init__(self, aes_key, iv, stream):
        LOGGER.debug('initializing encryptor')
        self._encryptor = Cipher(
            algorithms.AES(aes_key),
            modes.GCM(iv.encode('utf-8')),
            backend=default_backend(),
        ).encryptor()
        self._encryptor.authenticate_additional_data(b'')

        self.tag = None
        self._stream = stream

    def read(self, n_bytes=None):
        if self.tag is not None:
            return b''

        data = self._stream.read(n_bytes)
        encrypted_piece = self._encryptor.update(data)

        is_eof = (n_bytes is None) or (len(data) < n_bytes)
        if is_eof:
            LOGGER.debug('finalizing encryptor')
            encrypted_piece += self._encryptor.finalize()
            self.tag = self._encryptor.tag

        return encrypted_piece
