import os

import cars.settings
from cars.core.util import import_class
from .client import ObjectDoesNotExist


class MDSDocumentsWrapper:
    """
    MDS client wrapper for the needs of Carsharing registration.
    """

    ObjectDoesNotExist = ObjectDoesNotExist

    def __init__(self, client, bucket, aes_key):
        self._client = client
        self._bucket = bucket
        self._aes_key = aes_key

    @classmethod
    def from_settings(cls):
        client_class = import_class(cars.settings.MDS['encrypted_client_class'])
        return cls(
            client=client_class.from_settings(),
            bucket=cars.settings.USERS['documents']['mds']['bucket_name'],
            aes_key=cars.settings.USERS['documents']['mds']['aes_key'],
        )

    def get_user_document_photo(self, user_document_photo, image_processor=None):
        key = self._get_user_document_photo_key(user_document_photo)
        content = self._client.get_object_content(bucket=self._bucket, key=key)
        if image_processor is not None:
            content = image_processor.process(content)
        return content

    def put_user_document_photo(self, user_document_photo, content):
        key = self._get_user_document_photo_key(user_document_photo)
        return self._client.put_object(body=content, bucket=self._bucket, key=key)

    def get_user_document_background_video(self, user_document_background_video):
        key = self._get_user_document_background_video_key(user_document_background_video)
        return self._client.get_object(bucket=self._bucket, key=key)

    def put_user_document_background_video(self, user_document_background_video, stream):
        key = self._get_user_document_background_video_key(user_document_background_video)
        return self._client.upload_fileobj(fileobj=stream, bucket=self._bucket, key=key)

    def _get_user_document_photo_key(self, user_document_photo):
        if user_document_photo.document_id:
            return os.path.join(
                str(abs(user_document_photo.user.uid)),
                str(user_document_photo.document_id),
                str(user_document_photo.id),
            )
        else:
            return os.path.join(
                str(user_document_photo.user_id),
                str(user_document_photo.id)
            )

    def _get_user_document_background_video_key(self, user_document_background_video):
        if user_document_background_video.photo.document_id:
            return os.path.join(
                str(abs(user_document_background_video.photo.user.uid)),
                str(user_document_background_video.photo.document_id),
                str(user_document_background_video.id),
            )
        else:
            return os.path.join(
                str(user_document_background_video.photo.user_id),
                str(user_document_background_video.id)
            ) + '_bv'
