import copy
import enum
import logging

import furl

from cars.aggregator.static_data.car_models import CAR_MODELS
from cars.aggregator.static_data.car_photos import get_car_photo
from cars.aggregator.static_data.operators import LOCAL_OPERATORS


LOGGER = logging.getLogger(__name__)


class CarColor(enum.Enum):
    BLACK = 'Чёрный'
    BLACK_WHITE = 'Чёрно-белый'
    BLUE = 'Синий'
    GRAY = 'Серый'
    RED = 'Красный'
    WHITE = 'Белый'
    YELLOW = 'Жёлтый'


class CarTransmission(enum.Enum):
    AUTOMATIC = 'Автомат'
    MANUAL = 'Ручная'
    SEQUENTIAL = 'Секвентальная'


class Car(object):

    COLOR = CarColor
    TRANSMISSION = CarTransmission


    def __init__(self, id_, city_id, local_id, address_en, address_ru,
                 color, discount, fuel, is_free, lat, lon, operator, model,
                 plate_number, parking_tariff, usage_tariff, transmission, updated_at):
        self.id = id_
        self.city_id = city_id
        self.local_id = local_id
        self.address_en = address_en
        self.address_ru = address_ru
        self.color = color
        self.discount = discount
        self.fuel = fuel
        self.is_free = is_free
        self.lat = lat
        self.lon = lon
        self.operator = operator
        self.model = model
        self.plate_number = plate_number
        self.parking_tariff = parking_tariff
        self.usage_tariff = usage_tariff
        self.transmission = transmission
        self.updated_at = updated_at

    @property
    def address(self):
        return self.address_ru

    def localized_address(self, language):
        if language == 'ru':
            address = self.address_ru
        else:
            address = self.address_en
        return address

    def get_deeplink(self, scheme_only=False):
        operator = self.operator_obj
        if not operator:
            return ''

        deeplink_template = operator.deeplink
        if not deeplink_template:
            return ''

        try:
            deeplink = deeplink_template.format(car_id=self.local_id)
        except Exception:
            deeplink = ''
            LOGGER.exception('Failed to construct deeplink')

        # Hack to avoid full deeplinks for clients that don't support them.
        if deeplink and scheme_only:
            scheme = furl.furl(deeplink).scheme
            deeplink = '{}://'.format(scheme)

        return deeplink

    @property
    def operator_obj(self):
        return LOCAL_OPERATORS.get(self.city_id, {}).get(self.operator)

    @property
    def model_obj(self):
        return CAR_MODELS.get(self.model)

    @property
    def photo(self):
        return get_car_photo(operator=self.operator, model=self.model, color=self.color)

    def to_dict(self):
        data = copy.deepcopy(vars(self))
        data.pop('updated_at')
        data['updated_at_ts'] = self.updated_at.timestamp()
        return data
