class Operator(object):
    '''Generic operator representation'''

    def __init__(self, id_, short_name, color, website,
                 ios_store_url, android_store_url, faq_url, deeplink, icons):
        self.id = id_
        self.short_name = short_name
        self.color = color
        self.website = website
        self.ios_store_url = ios_store_url
        self.android_store_url = android_store_url
        self.faq_url = faq_url
        self.deeplink = deeplink  # String templated by {car_id}.
        self.icons = icons

    def __hash__(self):
        return hash(self.short_name)

    def __eq__(self, other):
        return isinstance(other, Operator) and self.short_name == other.short_name

    def __ne__(self, other):
        return not self == other

    def __repr__(self):
        return '<{}: {}>'.format(self.__class__.__name__, self.short_name)

    @property
    def _localization_key_prefix(self):
        return 'operator.{}'.format(self.short_name)

    def localized_name(self, translation):
        name_key = '{}.name'.format(self._localization_key_prefix)
        return translation.gettext(name_key)

    def localize(self, city, currency, phone, car_models, parkings):
        return LocalOperator(
            id_=self.id,
            short_name=self.short_name,
            color=self.color,
            website=self.website,
            ios_store_url=self.ios_store_url,
            android_store_url=self.android_store_url,
            faq_url=self.faq_url,
            deeplink=self.deeplink,
            icons=self.icons,
            city=city,
            currency=currency,
            phone=phone,
            car_models=car_models,
            parkings=parkings,
        )


class LocalOperator(Operator):
    '''City-dependent operator respresentation'''

    def __init__(self, city, currency, phone, car_models, parkings, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.city = city
        self.currency = currency
        self.phone = phone
        self.car_models = car_models
        self.parkings = parkings

    def __hash__(self):
        return hash(self.short_name) ^ hash(self.city)

    def __eq__(self, other):
        return (
            isinstance(other, LocalOperator)
            and self.short_name == other.short_name
            and self.city == other.city
        )

    def localized_info(self, translation):
        info_key = '{}.info.{}'.format(self._localization_key_prefix, self.city.short_name)
        return translation.gettext(info_key)


class OperatorColor(object):

    def __init__(self, red, green, blue):
        self.red = red
        self.green = green
        self.blue = blue

    def __repr__(self):
        return '<OperatorColor: {}>'.format(self.hex)

    @property
    def hex(self):
        return '#{:02x}{:02x}{:02x}'.format(
            *[int(255 * color) for color in [self.red, self.green, self.blue]]
        )


class OperatorIcons(object):

    def __init__(self, map_default, map_selected, map_zoom):
        self.map_default = map_default
        self.map_selected = map_selected
        self.map_zoom = map_zoom
