import furl
import logging
import requests

import kubiki.util

from cars.core.util import import_class
from cars.settings import NEW_BILLING as settings


LOGGER = logging.getLogger(__name__)


class NewBillingClient:

    def __init__(self, *, root_url, link_card_url, author_user_id, author_user_token,
                 default_url_args=None, default_headers=None,  timeout=10,
                 verify_ssl=False):
        self._author_user_id = str(author_user_id)
        self._author_user_token = author_user_token

        self._charge_furl = furl.furl(root_url).join('ticket')
        self._info_furl = furl.furl(root_url).join('info')
        self._link_card_furl = furl.furl(link_card_url)

        if default_url_args:
            for arg, val in default_url_args.items():
                self._charge_furl.args[arg] = val
                self._info_furl.args[arg] = val
                # self._link_card_furl.args[arg] = val  # no default args are needed

        self._default_headers = (
            {} if default_headers is None
            else default_headers
        )

        self._default_headers['Authorization'] = 'OAuth {}'.format(author_user_token)

        self._session = kubiki.util.make_requests_session()

        self._timeout = timeout
        self._verify_ssl = verify_ssl

    @classmethod
    def from_settings(cls):
        billing_client_class = import_class(settings['client_class'])
        return billing_client_class(
            root_url=settings['root_url'],
            link_card_url=settings['link_card_url'],
            default_url_args=settings['default_url_args'],
            default_headers=settings['default_headers'],
            author_user_id=settings['author_user_id'],
            author_user_token=settings['author_user_token'],
        )

    def _request(self, *, url, headers, method, data=None):
        response = self._session.request(
            method=method,
            url=url,
            headers=headers,
            json=data,
            timeout=self._timeout,
            verify=self._verify_ssl,
        )
        try:
            response.raise_for_status()
        except requests.exceptions.HTTPError:
            LOGGER.exception(
                'billing client request has bad status. Response: %s',
                response.content
            )
            raise
        return response

    def link_card(self, *, user_id, paymethod_id):
        if not paymethod_id:
            raise Exception('cannot link card: empty paymethod id for user {}'.format(user_id))
        url = self._link_card_furl.copy()
        headers = self._default_headers.copy()
        headers['UserIdDelegation'] = str(user_id)
        data = {
            'paymethod_id': paymethod_id,
        }
        self._request(url=url, headers=headers, data=data, method='POST')

    def charge(self, *, user_id, comment, sum, charge_type, session_id=None):
        url = self._charge_furl.copy()
        url.args['user_id'] = self._author_user_id
        if session_id is not None:
            url.args['session_id'] = str(session_id)
        headers = self._default_headers.copy()
        data = {
            'user_id': str(user_id),
            'sum': sum,
            'comment': comment,
            'session_id': str(session_id),
            'type': charge_type,
        }
        self._request(url=url, headers=headers, data=data, method='POST')

    def get_user_payments(self, *, user_id):
        url = self._info_furl.copy()
        url.args['user_id'] = str(user_id)
        headers = self._default_headers.copy()
        resp = self._request(url=url, headers=headers, method='GET')
        return resp.json()

    def get_payment_history(self, *, session_id, user_id):
        data = self.get_user_payments(user_id=user_id)
        history = data.get('history')
        if history is None:
            history = []
        return history

    def charge_exists(self, *, session_id=None, user_id=None, payment_history=None):
        if payment_history is None:
            payment_history = self.get_payment_history(
                session_id=session_id,
                user_id=user_id
            )
        return any(
            x['session_id'] == str(session_id)
            for x in payment_history
        )

    def charge_is_passed(self, *, session_id=None, user_id=None, payment_history=None):
        if payment_history is None:
            payment_history = self.get_payment_history(
                session_id=session_id,
                user_id=user_id
            )
        return any(
            (x['session_id'] == str(session_id)
             and x['state'] == 'finished'
             and x['task_status'] == 'ok')
            for x in payment_history
        )
