import copy
import logging
import retry
import urllib3
import uuid

import furl

from kubiki.util import make_requests_session

import cars.settings


LOGGER = logging.getLogger(__name__)


class Octopus(object):

    def __init__(self, url, api_key, verify_ssl=True):
        self._furl = furl.furl(url)
        self._api_key = api_key
        self._verify_ssl = verify_ssl
        self._session = make_requests_session()

    @property
    def furl(self):
        return self._furl.copy()

    def get_create_session_url(self):
        return self.furl.set(path='/api/session/create').url

    def get_session_log_url(self, session_id):
        path = '/api/session_log/query'
        args = {
            'session_id': session_id,
        }
        return self.furl.set(path=path, args=args).url

    @classmethod
    def from_settings(cls):
        octopus_settings = cars.settings.OCTOPUS
        return cls(
            url=octopus_settings['url'],
            api_key=octopus_settings['api_key'],
            verify_ssl=octopus_settings['verify_ssl'],
        )

    @retry.retry(
        exceptions=Exception,
        tries=3,
        delay=1
    )
    def _request(self, method, url, data=None):
        headers = {
            'Auth-Key': self._api_key,
        }
        response = self._session.request(
            method=method,
            url=url,
            headers=headers,
            json=data,
            verify=self._verify_ssl,
        )
        response.raise_for_status()
        return response

    def create_session(self, script):
        data = {
            'script': script,
        }
        response = self._request(
            method='POST',
            url=self.get_create_session_url(),
            data=data,
        )
        return response.json()['session_id']

    def get_session_log(self, session_id):
        return self._request(
            method='GET',
            url=self.get_session_log_url(session_id),
        ).json()

    def prepare(self, script):
        return PreparedOctopus(octopus_client=self, script=script)


class PreparedOctopus(object):

    def __init__(self, octopus_client, script):
        self._client = octopus_client
        self._script = script

    def call(self, number):
        LOGGER.info('calling %s', number)
        script = copy.deepcopy(self._script)
        script['steps']['start']['parameters']['called_number'] = number
        session_id = self._client.create_session(script=script)
        return session_id


class StubOctopus(object):

    @classmethod
    def from_settings(cls):
        return cls()

    def prepare(self, script):  # pylint: disable=unused-argument
        return StubPreparedOctopus()


class StubPreparedOctopus(object):

    def call(self, number):
        LOGGER.info('fake calling %s', number)
        return str(uuid.uuid4())


class StubOctopusFailing(object):

    @classmethod
    def from_settings(cls):
        return cls()

    def prepare(self, script):  # pylint: disable=unused-argument
        return StubPreparedOctopusFailing()


class StubPreparedOctopusFailing(object):

    def call(self, number):
        raise urllib3.exceptions.HTTPError('octopus fails')
