import os

import shapely.geometry
import shapely.wkt


STATIC_ROOT = os.path.join(os.path.dirname(__file__), 'static')


class ParkingAreasManager(object):

    DEFAULT_AREA = 'default'

    def __init__(self):
        self._areas = {}  # Short name -> MultiPolygon

    def get_area(self, date_at=None, format_='geojson'):  # pylint: disable=unused-argument
        return self._get(self.DEFAULT_AREA, format_=format_)

    def _get(self, name, format_='geojson'):
        shape = self._areas[name]
        if format_ == 'geojson':
            area = shapely.geometry.mapping(shape)
        else:
            raise RuntimeError('Unsupported area format: {}'.format(format_))
        return area

    def load(self, name, area):
        return self.load_union(name, [area])

    def load_union(self, name, areas):
        result = shapely.geometry.Polygon()

        for area in areas:
            polygon = self._load_raw(area)
            result = result.union(polygon)

        # Ensure the resulting type is a MultiPolygon.
        if isinstance(result, shapely.geometry.Polygon):
            result = shapely.geometry.MultiPolygon([result])

        self._areas[name] = result

    def _load_raw(self, filename):
        full_filename = '{}.wkt'.format(filename)
        path = os.path.join(STATIC_ROOT, full_filename)
        with open(path) as f:
            area = shapely.wkt.load(f)
        return area


class AnyTimeParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='anytime')


class BelkaCarParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='belka')


class Car5ParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='car5')


class ColesaSpbParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='colesa_spb')


class DelimobilMskParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load_union(
            name=self.DEFAULT_AREA,
            areas=['mkad', 'metro_stations_1km', 'delimobil_tc', 'delimobil_airports'],
        )

class DelimobilSpbParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='delimobil_spb')


class DriveTimeParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='drivetime')


class EasyRideParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='mkad')


class GreenGoParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load_union(
            name=self.DEFAULT_AREA,
            areas=['greengo_budapest', 'greengo_margitsziget'],
        )


class MaturCarUfaParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='maturcar_ufa')


class RentmeeMskParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='rentmee_msk')


class RentmeeSpbParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='rentmee_spb')


class URentCarAnapaParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='urentcar_anapa')

class URentCarCmwParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='urentcar_cmw')

class URentCarKrasnodarParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='urentcar_krasnodar')

class URentCarNovorossiyskParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='urentcar_novorossiysk')

class URentCarSochiParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='urentcar_sochi')


class YandexDriveMskParkingAreasManager(ParkingAreasManager):
    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.DEFAULT_AREA, area='yandex_drive_msk')


class YouDriveParkingAreasManager(ParkingAreasManager):

    WEEKDAY_AREA = 'weekday'
    WEEKEND_AREA = 'weekend'

    def get_area(self, date_at=None, format_='geojson'):
        assert date_at is not None
        if date_at.weekday() < 5:
            area = self.WEEKDAY_AREA
        else:
            area = self.WEEKEND_AREA
        return self._get(area, format_=format_)


class YouDriveMskParkingAreasManager(YouDriveParkingAreasManager):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.WEEKDAY_AREA, area='youdrive_specific_weekday_msk')
        self.load_union(
            name=self.WEEKEND_AREA,
            areas=['youdrive_specific_weekend_msk', 'mkad'],
        )


class YouDriveSpbParkingAreasManager(YouDriveParkingAreasManager):

    def __init__(self, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.load(name=self.WEEKDAY_AREA, area='youdrive_specific_weekday_spb')
        self.load(name=self.WEEKEND_AREA, area='youdrive_specific_weekend_spb')
