import collections
import datetime
import itertools
import logging

from kubiki.util import make_requests_session

import cars.settings
from cars.core.tvm2 import TVM2ServiceTicket


LOGGER = logging.getLogger(__name__)


class SocialAccountInfoManager(object):
    BBSocialAccountInfo = collections.namedtuple(
        'BBSocialAccountInfo', ('profile_id', 'code', 'name', 'display_name')
    )
    SocialApiAccountInfo = collections.namedtuple(
        'SocialApiAccountInfo', ('profile_id', 'profile_addresses', 'user_id', 'username',
                                 'provider_code', 'provider_name', 'provider_display_name', )
    )

    # full list: https://wiki.yandex-team.ru/social/providers/
    social_provider_mapping = {
        'vk': {'display_name': 'ВКонтакте', 'name': 'vkontakte'},
        'fb': {'display_name': 'Facebook', 'name': 'facebook'},
        'tw': {'display_name': 'Twitter', 'name': 'twitter'},
        'mr': {'display_name': 'Mail.ru', 'name': 'mailru'},
        'gg': {'display_name': 'Google', 'name': 'google'},
        'ok': {'display_name': 'Одноклассники', 'name': 'odnoklassniki'},
    }

    @classmethod
    def from_bb_social_account_info(cls, profile_id, provider, **kwargs):
        # refer to https://wiki.yandex-team.ru/passport/social/about/#kakotobrazhatpolzovateljavshapkeikommentarijax
        # profile_id, redirect_target, provider (code)
        extra_provider_info = cls.social_provider_mapping.get(provider, {})
        account_info = cls.BBSocialAccountInfo(
            profile_id=profile_id,
            code=provider,
            name=extra_provider_info.get('name', provider),
            display_name=extra_provider_info.get('display_name', provider),
        )
        return account_info

    @classmethod
    def from_social_api_account_info(cls, profile_id, addresses, provider, provider_code, userid, username, **kwargs):
        # refer to https://wiki.yandex-team.ru/social/api/
        # profile_id, addresses, provider (name), provider_code, uid, userid (ext. social), username (ext. social)
        extra_provider_info = cls.social_provider_mapping.get(provider_code, {})
        account_info = cls.SocialApiAccountInfo(
            profile_id=profile_id,
            profile_addresses=addresses,
            user_id=userid,
            username=username,
            provider_code=provider_code,
            provider_name=provider,
            provider_display_name=extra_provider_info.get('display_name', provider),
        )
        return account_info


class SocialInfoClient(object):
    def __init__(self, base_url, ticket, timeout=1):
        self._base_url = base_url
        self._ticket = ticket
        self._session = make_requests_session()
        self._timeout = timeout

    @classmethod
    def from_settings(cls):
        base_url = cars.settings.SOCIAL['url'] + cars.settings.SOCIAL['profile_list_url']
        ticket = TVM2ServiceTicket(**cars.settings.SOCIAL['tvm2'])
        timeout = cars.settings.SOCIAL['timeout']
        return cls(base_url=base_url, ticket=ticket, timeout=timeout)

    def get_account_info(self, uid):
        response = None
        user_profiles = []

        try:
            response = self._session.get(
                self._get_request_url(uid),
                headers=self._get_request_headers(),
                json=self._get_request_json_data(),
                timeout=self._timeout,
                verify=False
            )
            response.raise_for_status()

            profiles_data = response.json().get('profiles', [])

            user_profiles = [
                SocialAccountInfoManager.from_social_api_account_info(**profile_data)._asdict()
                for profile_data in profiles_data
            ]

        except Exception:
            response_content = getattr(response, 'text', '')
            LOGGER.exception('unable to load social account data, response content: {}'.format(response_content))

        return user_profiles

    def _get_request_url(self, uid):
        uid = str(uid).lstrip('-')  # process drive deleted accounts correctly
        request_url = self._base_url + '?uid={}'.format(uid)
        return request_url

    def _get_request_headers(self):
        headers = {'X-Ya-Service-Ticket': self._ticket.content}
        return headers

    def _get_request_json_data(self):
        data = {'consumer': cars.settings.SOCIAL['consumer']}
        return data


class AccountData(object):
    social_info_client = SocialInfoClient.from_settings()

    # attributes and aliases description: https://wiki.yandex-team.ru/passport/dbmoving
    PUBLIC_NAME_ATTR_ID = '1007'
    LOGIN_ATTR_ID = '1008'
    IS_AVAILABLE_ATTR_ID = '1009'
    PLUS_STATUS_ATTR_ID = '1015'

    SOCIAL_ALIAS_ID = '6'
    PDD_ALIAS_ID = '7'

    def __init__(self, user_info=None, *, is_erroneous=False):
        user_info = user_info or {}

        # user info description: https://docs.yandex-team.ru/blackbox/methods/userinfo#response_format
        self._user_info = user_info
        self._is_erroneous = is_erroneous

        self._fields_mapping = {k: v for k, v in user_info.get('dbfields', {}).items() if v != ''}
        self._attributes = {k: v for k, v in user_info.get('attributes', {}).items() if v}

        self._social_profiles = None

    @classmethod
    def iter_public_attribute_ids(cls):
        return (v for k, v in vars(cls).items() if k.endswith('_ATTR_ID') and isinstance(v, str))

    @property
    def is_erroneous(self):
        return self._is_erroneous

    def get_property_safe(self, property_name, *, default=None):
        return getattr(self, property_name) if not self.is_erroneous else default

    def has_attribute(self, attribute_id):
        return self._attributes.get(attribute_id, '0') == '1'

    @property
    def exists(self):
        return self.uid is not None

    # raw user info

    @property
    def raw_login(self):
        return self._user_info.get('login', None)

    @property
    def has_password(self):
        return self._user_info.get('have_password', None)

    @property
    def is_social(self):
        return not self.raw_login and not self.has_password

    @property
    def uid(self):
        return self._user_info.get('uid', {}).get('value', None)

    @property
    def is_hosted(self):
        return self._user_info.get('uid', {}).get('hosted', False)  # pdd (external domain) account

    @property
    def external_domain(self):
        return self._user_info.get('uid', {}).get('domain', None)  # pdd (external domain) account

    @property
    def display_name(self):
        return self._user_info.get('display_name', {}).get('name', None)

    @property
    def social_account_info(self):
        social_data = self._user_info.get('display_name', {}).get('social', {})
        if social_data:
            social_account_info = SocialAccountInfoManager.from_bb_social_account_info(**social_data)._asdict()
        else:
            social_account_info = None
        return social_account_info

    # attribute info

    @property
    def plus_status(self):
        return self.has_attribute(self.PLUS_STATUS_ATTR_ID)

    @property
    def public_name(self):
        return self._attributes.get(self.PUBLIC_NAME_ATTR_ID, None)

    @property
    def login(self):
        return self._attributes.get(self.LOGIN_ATTR_ID, None)

    @property
    def is_available(self):
        return self.has_attribute(self.IS_AVAILABLE_ATTR_ID)

    # aliases info

    @property
    def aliases(self):
        return self._user_info.get('aliases', {})

    @property
    def is_social_by_attr(self):
        return self.SOCIAL_ALIAS_ID in self.aliases

    @property
    def is_hosted_by_attr(self):
        return self.PDD_ALIAS_ID in self.aliases

    # fields info

    @property
    def num_subscriptions_weighted(self):
        num_subscriptions_weighted = sum(
            weight for field, weight in cars.settings.BLACKBOX['suid_weights'].items()
            if field in self._fields_mapping
        )
        return num_subscriptions_weighted

    @property
    def account_age(self):
        account_age = datetime.timedelta()

        reg_date_str = self._fields_mapping.get('account_info.reg_date.uid', '')[:10]

        if reg_date_str:
            reg_date = datetime.datetime.strptime(reg_date_str, '%Y-%m-%d').date()
            account_age = datetime.date.today() - reg_date

        return account_age

    @property
    def seeds(self):
        return [x.split('.')[-1] for x in self._fields_mapping if x.startswith('subscription.suid')]

    # extra info

    @property
    def social_profiles_info(self):
        if self._social_profiles is None and self.uid is not None:
            self._social_profiles = self.social_info_client.get_account_info(self.uid)
        return self._social_profiles or []


class PassportClient(object):
    not_set = object()

    def __init__(self, base_url, ticket=None, timeout=1):
        self._base_url = base_url
        self._ticket = ticket
        self._session = make_requests_session()
        self._timeout = timeout

    @classmethod
    def from_settings(cls, *, bb_url_section='url', tvm_section='tvm', timeout=1):
        base_url = cars.settings.BLACKBOX[bb_url_section]

        if tvm_section is not None:
            ticket = TVM2ServiceTicket(**cars.settings.BLACKBOX[tvm_section])
        else:
            ticket = None  # case to not use tvm ticket at all (used to obtain plus status on prod api)

        return cls(base_url=base_url, ticket=ticket, timeout=timeout)

    @classmethod
    def from_settings_admin_api(cls, *, bb_url_section='url_external', tvm_section=not_set, timeout=1):
        # use external blackbox and corresponding tvm ticket (used on admin api)
        if tvm_section is cls.not_set:
            tvm_section = cars.settings.BLACKBOX['default_tvm_section']

        return cls.from_settings(
            bb_url_section=bb_url_section,
            tvm_section=tvm_section,
            timeout=timeout
        )

    def do_account_exists(self, uid, default=None):
        account_info = self.get_account_info(uid)
        exists = account_info.get_property_safe('exists', default=default)
        return exists

    def get_plus_status(self, uid, default=None):
        account_info = self.get_account_info(uid, attributes=[AccountData.PLUS_STATUS_ATTR_ID])
        plus_status = account_info.get_property_safe('plus_status', default=default)
        return plus_status

    def get_account_public_info(self, uid, db_fields=(), attributes=()):
        attributes = tuple(itertools.chain(attributes, AccountData.iter_public_attribute_ids()))
        return self.get_account_info(uid, db_fields, attributes)

    def get_account_info(self, uid, db_fields=(), attributes=()):
        response = None

        try:
            response = self._session.get(
                self._get_request_url(uid, db_fields, attributes),
                headers=self._get_request_headers(),
                timeout=self._timeout
            )
            response.raise_for_status()

            user_info = response.json()['users'][0]

            account_data = AccountData(user_info)

        except Exception:
            account_data = AccountData(is_erroneous=True)

            response_content = getattr(response, 'text', '')
            LOGGER.exception('unable to load blackbox account data, response content: {}'.format(response_content))

        return account_data

    def _get_request_url(self, uid, db_fields, attributes):
        uid = str(uid).lstrip('-')  # process drive deleted accounts correctly

        url_parts = [
            self._base_url + '?method=userinfo&userip=8.8.8.8&user_ip=8.8.8.8&format=json',
            'regname=yes',
            'aliases=all',
            'uid={}'.format(uid),
            'dbfields={}'.format(','.join(db_fields)) if db_fields else '',
            'attributes={}'.format(','.join(attributes)) if attributes else '',
        ]

        request_url = '&'.join(x for x in url_parts if x)
        return request_url

    def _get_request_headers(self):
        headers = {}
        if self._ticket is not None:
            headers['X-Ya-Service-Ticket'] = self._ticket.content
        return headers
