import concurrent.futures
import logging

import shapely.geometry
from haversine import haversine

import saaspy.util.geo

import cars.settings


LOGGER = logging.getLogger(__name__)


class PedestrianRouter(object):

    SAAS_PERM = 'ptPedestrian'

    def __init__(self):
        config = cars.settings.PEDESTRIAN_ROUTER

        self._tie_radius = config['TIE_RADIUS']
        self._max_straight_distance = config['MAX_STRAIGHT_DISTANCE']
        self._walk_speed = config['WALK_SPEED']

        saas_config = config['SAAS_CONFIG']
        self._saas = saaspy.client.GeoSaasClient(**saas_config)

    def get_walk_time(self, p1, p2):
        p1 = saaspy.util.geo.Point(lon=p1[0], lat=p1[1])
        p2 = saaspy.util.geo.Point(lon=p2[0], lat=p2[1])

        straight_distance = haversine(p1, p2) * 1000

        if straight_distance > self._max_straight_distance:
            walk_time = straight_distance / self._walk_speed
        else:
            response = self._saas.get_route(
                from_point=p1,
                to_point=p2,
                perm=self.SAAS_PERM,
                d=self._tie_radius,
                search_props={
                    'sp_meta_search': None,
                },
            )
            if response.summary is None:
                LOGGER.error('No data from get_route: %s -> %s', p1, p2)
                walk_time = straight_distance / self._walk_speed
            else:
                walk_time = response.summary.time

        return walk_time

    def build_walk_area_from_distance(self, point, distance):
        saas_point = saaspy.util.geo.Point(lat=point[1], lon=point[0])
        response = self._saas.get_routing_area(start=saas_point,
                                               border=distance,
                                               perm=self.SAAS_PERM,
                                               v='1 1',
                                               d=150,
                                               summary_only=True,
                                               n_retries=1)
        area = self._build_walk_area_from_hull(response.area)
        return area

    def build_walk_area_from_time(self, point, walk_seconds):
        walk_distance = self._walk_speed * walk_seconds
        return self.build_walk_area_from_distance(point, walk_distance)

    def build_many_areas_from_distance(self, point, min_distance, max_distance, num_areas):
        step = float(max_distance - min_distance) / num_areas
        distances = [min_distance + i * step for i in range(num_areas)]
        points = [point] * num_areas

        pool = concurrent.futures.ThreadPoolExecutor(max_workers=num_areas)
        areas = list(pool.map(self.build_walk_area_from_distance, points, distances))
        pool.shutdown()

        for i in reversed(range(1, len(areas))):
            area = areas[i]
            smaller_area = areas[i-1]
            if not smaller_area or smaller_area.is_empty:
                areas[i-1] = area

        result = []
        for distance, area in zip(distances, areas):
            result.append({
                'distance': distance,
                'area': shapely.geometry.mapping(area) if not area.is_empty else None,
            })

        return result

    def build_many_areas_from_time(self, point, min_walk_seconds, max_walk_seconds, num_areas):
        min_walk_distance = self._walk_speed * min_walk_seconds
        max_walk_distance = self._walk_speed * max_walk_seconds
        result = self.build_many_areas_from_distance(
            point, min_walk_distance, max_walk_distance, num_areas
        )

        for item in result:
            distance = item.pop('distance')
            item['walk_time'] = distance / self._walk_speed

        return result

    def _build_walk_area_from_hull(self, saas_polygon):
        points = [(p.lon, p.lat) for p in saas_polygon.points]
        points = points or None  # points may be empty
        hull = shapely.geometry.Polygon(points)
        area = self._smooth_polygon(hull)
        return area

    def _smooth_polygon(self, polygon,
                        inital_buffer=0.0005, max_attempts=3, simpify_tolerance=0.00001):

        if polygon.is_empty:
            return polygon

        smoothed_polygon = polygon
        buffer_width = inital_buffer

        for _ in range(max_attempts):
            smoothed_polygon = (polygon
                                .buffer(-buffer_width, join_style=1)
                                .buffer(buffer_width, join_style=1))
            if not smoothed_polygon.is_empty:
                break
            buffer_width /= 2

        smoothed_polygon = smoothed_polygon.simplify(tolerance=simpify_tolerance)

        return smoothed_polygon
