import asyncio
import copy
import enum
import json
import logging
import retry
import time

import pymorphy2

import kubiki.xiva

import cars.settings
from cars.aggregator.serializers import CarSerializer
from cars.aggregator.static_data import CAR_MODELS, OPERATORS
from cars.core.constants import AppPlatform
from cars.core.l10n import get_translations
from cars.core.pedestrian_router import PedestrianRouter
from cars.core.push_client import CarsharingPushClientLogger
from cars.core.util import import_class
from cars.core.solomon import make_solomon_client
from cars.django.lock import UwsgiLock


LOGGER = logging.getLogger(__name__)


class Xiva(kubiki.xiva.Xiva):

    class PushStatus(enum.Enum):
        OK_200 = 200
        FILTERED_202 = 202
        NOT_SUBSCRIBED_204 = 204
        SUBSCRIPTION_REMOVED_205 = 205
        BAD_REQUEST_400 = 400
        FORBIDDEN_403 = 403
        DUPLICATE_IDENTIFIER_409 = 409
        RATE_LIMIT_ERROR_429 = 429
        TRANSPORT_ERROR_500 = 500
        PUSH_SERVICE_ERROR_502 = 502
        PUSH_SERVICE_TIMEOUT_504 = 504

    def __init__(self, *args, push_client=None, **kwargs):
        super().__init__(*args, **kwargs)
        self._push_client = push_client

    @classmethod
    def from_settings(cls, push_client=None):
        xiva_settings = cars.settings.XIVA
        xiva_client_class = import_class(xiva_settings['client_class'])
        if push_client is None:
            push_client = CarsharingPushClientLogger(
                filename=xiva_settings['push_client']['filename'],
                lock=UwsgiLock(),
            )
        return xiva_client_class(
            url=xiva_settings['url'],
            service=xiva_settings['service'],
            send_token=xiva_settings['send_token'],
            subscription_token=xiva_settings['subscription_token'],
            push_client=push_client,
        )

    def on_response(self, response):
        super().on_response(response)
        self._log_response(response)

    def _log_response(self, response):
        if self._push_client is None:
            return

        request = response.request

        request_data = request.body if request.body else None
        if isinstance(request_data, bytes):
            request_data = request_data.decode('utf-8')

        data = {
            'method': request.method,
            'url': request.url,
            'data': request_data,
            'status_code': response.status_code,
            'response': response.text,
            'response_headers': dict(response.headers),
        }

        self._push_client.log(
            type_='xiva.reqans',
            data=data,
        )

    def get_send_batch_url(self, send_token, event, ttl=None):
        path = '/v2/batch_send'
        args = {
            'token': send_token,
            'event': event,
        }
        if ttl is not None:
            args['ttl'] = ttl
        url = self._urls._furl.set(path=path, args=args).url
        return url

    def send_batch(self, event, payload, users, ttl=None, apns=None, gcm=None):
        url = self.get_send_batch_url(
            send_token=self._send_token,
            event=event,
            ttl=ttl
        )

        data = {
            'recipients': [str(u) for u in users],
            'payload': payload,
        }
        if apns:
            repack = data.setdefault('repack', {})
            repack['apns'] = apns.to_payload()
        if gcm:
            repack = data.setdefault('repack', {})
            repack['gcm'] = gcm.to_payload()

        data_json = json.dumps(data, ensure_ascii=False)
        LOGGER.info('sending push to batch of urls; url = `%s`, data_json = `%s`',
                    url, data_json)

        return self._post(url, data=data_json.encode('utf-8'))


class BasePusher(object):
    def __init__(self, xiva_client, android_app_name, ios_app_name, solomon_client=None, push_result_reporter=None):
        self._xiva = xiva_client
        self._android_app_name = android_app_name
        self._ios_app_name = ios_app_name
        self._solomon_client = solomon_client
        self._push_result_reporter = push_result_reporter

    @classmethod
    def from_settings(cls, push_client=None, solomon_client=None):
        xiva_client = Xiva.from_settings(
            push_client=push_client,
        )

        if solomon_client is None:
            if cars.settings.SOLOMON is None:
                solomon_config = None
            else:
                solomon_config = copy.deepcopy(cars.settings.SOLOMON)
                solomon_config['service'] = cars.settings.XIVA['solomon']['service']
            solomon_client = make_solomon_client(solomon_config)

        push_result_reporter_cls = import_class(cars.settings.SEND_PUSH['reporter']['client_class'])
        push_result_reporter = push_result_reporter_cls()

        return cls(
            xiva_client=xiva_client,
            android_app_name=cars.settings.XIVA[AppPlatform.ANDROID]['app_name'],
            ios_app_name=cars.settings.XIVA[AppPlatform.IOS]['app_name'],
            solomon_client=solomon_client,
            push_result_reporter=push_result_reporter,
        )

    def get_xiva_codes_from_user_result(self, result):
        if isinstance(result, dict):
            return [Xiva.PushStatus(result['code'])]
        elif isinstance(result, list):
            return [Xiva.PushStatus(r['code']) for r in result]
        else:
            raise ValueError('unknown response type: {}'.format(result))

    def send(self, uid, message=None, payload=None, sender=None):
        if payload is None:
            payload = {}

        if message is None:
            apns_meta = None
            gcm_meta = None
        else:
            apns_meta = kubiki.xiva.ApnsMeta(
                alert=message,
                sound='default',
                badge=1,
            )
            gcm_meta = kubiki.xiva.GcmMeta(
                notification=kubiki.xiva.GcmNotificationMeta(body=message),
            )

        response = self._xiva.send(
            user=str(uid),
            payload=payload,
            apns=apns_meta,
            gcm=gcm_meta,
            event='push',
            delivery_mode=self._xiva.DeliveryMode.DIRECT,
        )

        self._monitor('send.http_code', 1, labels={'http_code': response.status_code})

        try:
            response_data = response.json()
            codes = self.get_xiva_codes_from_user_result(response_data)

            for code in codes:
                self._monitor('send.code', 1, labels={'code': code.value})
                if code != Xiva.PushStatus.OK_200:
                    LOGGER.error('unexpected xiva response: %s', response.json())

            self._push_result_reporter.report(message, uid, codes, sender)

        except Exception:
            LOGGER.exception('failed to parse xiva response: %s', response.text)

    @asyncio.coroutine
    def _send_limited_batch(self, *, uids, message, payload, ttl, event):
        if payload is None:
            payload = {}

        if message is None:
            apns_meta = None
            gcm_meta = None
        else:
            apns_meta = kubiki.xiva.ApnsMeta(
                alert=message,
                sound='default',
                badge=1,
            )
            gcm_meta = kubiki.xiva.GcmMeta(
                notification=kubiki.xiva.GcmNotificationMeta(body=message),
            )

        response = self._xiva.send_batch(
            users=uids,
            payload=payload,
            ttl=ttl,
            apns=apns_meta,
            gcm=gcm_meta,
            event=event,
        )
        return zip(uids, response.json()['results'])

    def send_batch(self, *, uids, message=None, payload=None, ttl=None,
                   event='carsharing_push', sender=None):
        max_chunk_size = 10000
        uids_chunks = [uids[i:i+max_chunk_size]
                       for i in range(0, len(uids), max_chunk_size)]
        tasks = [
            asyncio.async(self._send_limited_batch(
                uids=uids_chunk,
                message=message,
                payload=payload,
                ttl=ttl,
                event=event,
            ))
            for uids_chunk in uids_chunks
        ]
        loop = asyncio.get_event_loop()

        task_results = loop.run_until_complete(asyncio.wait(tasks))[0]

        result_mapping = dict.fromkeys(uids)
        result_codes_mapping = dict.fromkeys(uids)

        for task_result in task_results:
            if task_result.exception() is not None:
                try:
                    task_result.result()
                except Exception:
                    LOGGER.exception('Failed to send batch of pushes')
                    continue

            for uid, uid_res in task_result.result():
                result_mapping[uid] = uid_res
                result_codes_mapping[uid] = self.get_xiva_codes_from_user_result(uid_res)

        self._push_result_reporter.report_batch(message, result_codes_mapping, sender)

        return result_mapping

    def subscribe(self, uid, uuid, platform, app_name, push_token):
        xiva_platform = self._get_xiva_platform(platform=platform)
        self._xiva.subscribe_app(
            app_name=app_name,
            platform=xiva_platform,
            user=str(uid),
            uuid=str(uuid),
            push_token=push_token,
        )

        LOGGER.info('subscribed user %s to pushes: platform=%s uuid=%s', uid, platform, uuid)

        self._monitor(
            'subscribe.count',
            1,
            labels={
                'app_name': app_name,
                'platform': xiva_platform,
            },
        )

    def _get_xiva_platform(self, platform):
        if platform is AppPlatform.ANDROID:
            xiva_platform = 'gcm'
        elif platform is AppPlatform.IOS:
            xiva_platform = 'apns'
        else:
            raise RuntimeError('unreachable: {}'.format(platform))
        return xiva_platform

    def _monitor(self, subtype, value, labels=None):
        if self._solomon_client is None:
            return
        sensor = 'xiva.{}'.format(subtype)
        self._solomon_client.set_value(sensor, value, labels=labels)


class Pusher(object):
    """Aggregator push manager"""

    def __init__(self, push_client=None):
        self._app_name = cars.settings.XIVA[AppPlatform.IOS]['app_name']
        self._xiva = Xiva.from_settings(push_client=push_client)
        self._prouter = PedestrianRouter()
        self._text_maker = TextMaker()

    def send(self, platform, token, message, payload=None):
        user = self._get_xiva_user(platform=platform, token=token)
        self._subscribe(platform, user, token)
        self._send(user, message, payload=payload)

    def _subscribe(self, platform, user, token):
        self._xiva.subscribe_app(
            app_name=self._app_name, platform=platform, user=user, uuid='0', push_token=token,
        )

    def _send(self, user, message, payload=None):
        if payload is None:
            payload = {}
        apns_meta = kubiki.xiva.ApnsMeta(alert=message)
        self._xiva.send(user=user, event='greeting', payload=payload, apns=apns_meta)

    def send_free_car_push(self, platform, token, location, car):
        walk_time_min = self._get_walk_time_min(location.coords[0], (car.lon, car.lat))
        message = self._text_maker.make_new_car_message(car, walk_time_min)
        payload = {
            'car': CarSerializer(car).data,
        }
        self.send(platform, token, message, payload=payload)

    def send_reminder(self, platform, token, alert):
        searching_min = int((time.time() - alert.created_at) / 60)
        message = self._text_maker.make_reminder_message(searching_min)
        self.send(platform, token, message)

    def _get_xiva_user(self, platform, token):
        return '{}-{}'.format(platform, token)

    def _get_walk_time_min(self, p1, p2):
        walk_time = self._prouter.get_walk_time(p1, p2)
        walk_time_min = int(walk_time / 60)
        return walk_time_min


class TextMaker(object):

    def __init__(self):
        self._morph = pymorphy2.MorphAnalyzer()
        self._minute_word = self._morph.parse('минута')[0].inflect({'accs'})
        self._translations = get_translations('ru')

    def make_new_car_message(self, car, walk_time_min):
        operator = OPERATORS[car.operator]
        operator_name = operator.localized_name(self._translations)
        inflected_operator_name = self._inflect_operator_gent(operator_name)

        model = CAR_MODELS[car.model]
        model_name = model.localized_name(self._translations)

        minutes_str = self._agree_minute_with_number(walk_time_min)

        message = ('Нашлась машина «{}» {}.\n{} {} пешком.'
                   .format(inflected_operator_name, model_name, walk_time_min, minutes_str))

        return message

    def make_reminder_message(self, searching_min):
        minutes_str = self._agree_minute_with_number(searching_min)
        message = 'Ищем машину уже {} {}'.format(searching_min, minutes_str)
        return message

    def _agree_minute_with_number(self, number):
        return self._minute_word.make_agree_with_number(number).word

    def _inflect_operator_gent(self, operator_name):
        inflected = self._morph.parse(operator_name)[0].inflect({'gent'})
        if inflected:
            result = inflected.word.capitalize()
        else:
            result = operator_name
        return result
