import contextlib
import collections
import decimal
import logging
import math

import furl
import requests
import retry

from django_yauth.util import get_real_ip
from kubiki.util import make_requests_session

import cars.settings
from cars.core.util import Location


LOGGER = logging.getLogger(__name__)


OfferInfo = collections.namedtuple(
    'OfferInfo',
    [
        'vid',
        'access_type',
        'car_position',
        'destination',
        'distance_dest',
        'fix_price',
        'time_dest',
    ],
)


def convert_header_to_env_var(value):
    return 'HTTP_' + value.upper().replace('-', '_')


def try_convert_env_var_to_header(value):
    # some headers may have not leading uppercase letters (e.g. SecretVersion, UUID)
    if value.startswith('HTTP_'):
        value = value[len('HTTP_'):]
    return '-'.join(x.capitalize() for x in value.split('_'))


class SaasDrive:
    EXPLICIT_HEADERS = (
        'Accept-Encoding', 'Accept-Language', 'Content-Type', 'User-Agent', 'Timezone-Offset',
        'DeviceID', 'UUID', 'Lat', 'Lon', 'SecretVersion', 'Authorization', 'Cookie',
        'AC_AppBuild', 'AC_AdjustToken', 'IC_AppBuild', 'IC_AdjustToken', 'AS_AppBuild', 'IMEI',
        'AppVersion', 'AppBuild', 'BuildType', 'UserIdDelegation'
    )

    EXPLICIT_ENV_VAR_TO_HEADER_MAPPING = {convert_header_to_env_var(h): h for h in EXPLICIT_HEADERS}

    class OfferNotFoundError(Exception):
        pass

    def __init__(self, *, url, service, prestable, version, timeout,
                 default_headers=None):
        self._urls = SaasDriveUrls(root_url=url, service=service,
                                   prestable=prestable, version=version)
        self._timeout = timeout
        self._session = make_requests_session()
        self._default_headers = default_headers or {}

    @classmethod
    def from_settings(cls, **kwargs):
        settings = cars.settings.SAAS_DRIVE
        kw = {
            'url': settings['url'],
            'service': settings['service'],
            'prestable': settings['prestable'],
            'version': settings['version'],
            'timeout': settings['timeout'],
            'default_headers': settings['default_headers'],
        }
        kw.update(kwargs)
        return cls(**kw)

    @contextlib.contextmanager
    def with_custom_auth(self, request):
        assert request is not None, 'no authentication method has been provided'

        original_headers = self._session.headers.copy()
        original_cookies = self._session.cookies.copy()

        if hasattr(request, '_request'):
            original_request = getattr(request, '_request')
        else:
            original_request = request

        for h in self.EXPLICIT_HEADERS:
            self._session.headers.pop(h, None)

        for env_var_name, env_var_value in original_request.META.items():
            if env_var_name in self.EXPLICIT_ENV_VAR_TO_HEADER_MAPPING:
                self._session.headers[self.EXPLICIT_ENV_VAR_TO_HEADER_MAPPING[env_var_name]] = env_var_value
            elif env_var_name.startswith('HTTP_X_'):
                self._session.headers[try_convert_env_var_to_header(env_var_name)] = env_var_value

        user_ip = get_real_ip(original_request)
        if user_ip is not None:
            self._session.headers.setdefault('X-Forwarded-For', user_ip)
            self._session.headers.setdefault('X-Forwarded-For-Y', user_ip)

        for default_header_name, default_header_value in self._default_headers.items():  # particularly AppBuild
            self._session.headers.setdefault(default_header_name, default_header_value)

        try:
            yield self

        finally:
            self._session.cookies = original_cookies
            self._session.headers = original_headers

    def get_fines(self, request):
        with self.with_custom_auth(request):
            url = self._urls.get_fines_url()
            response = self._session.request('GET', url, timeout=self._timeout)
            try:
                response.raise_for_status()
            except requests.exceptions.HTTPError:
                LOGGER.exception(
                    'error getting fines. Response: %s',
                    response.content
                )
                raise
            return response.json()

    def get_offer_info(self, oid, oauth_token):
        url = self._urls.get_offer_info_url(oid=oid)
        try:
            response = self._request(url=url, token=oauth_token)
        except requests.HTTPError as e:
            if e.response is not None and e.response.status_code == 404:
                raise self.OfferNotFoundError
            raise

        data = response.json()['data']
        car_position_lon, car_position_lat = data['car_position']
        destination_lon, destination_lat = data['destination']
        offer_info = OfferInfo(
            vid=data['VID'],
            access_type=data['access_type'],
            car_position=Location(lat=car_position_lat, lon=car_position_lon),
            destination=Location(lat=destination_lat, lon=destination_lon),
            distance_dest=data['distance_dest'],
            fix_price=decimal.Decimal(data['fix_price']) / 100,
            time_dest=data['time_dest'],
        )

        return offer_info

    def get_tracks(self, order_id, oauth_token):
        url = self._urls.get_tracks_url(order_id=order_id)
        response = self._request(url=url, token=oauth_token, log=False).json()

        tracks = response.get('tracks')
        if not tracks:
            return set()

        result_tracks = set()
        for track in tracks:
            timestamps = map(int, track['timestamps'].split(' '))
            coords = [float(x) for x in track['coords'].split(' ') if x]
            longitudes = coords[::2]
            latitudes = coords[1::2]
            result_tracks.add(zip(timestamps, latitudes, longitudes))
        return result_tracks

    def get_nearest_session(self, timestamp, car_id, oauth_token):
        url = self._urls.get_nearest_session_url(timestamp, car_id)
        headers = {
        }
        try:
            return self._request(url=url, headers=headers, log=False, token=oauth_token).json()
        except requests.exceptions.HTTPError as e:
            if e.response.status_code == 404:
                return None
            raise

    def get_nearest_session_custom(self, timestamp, car_id, oauth_token, *, max_skip=4, min_mileage=0.1):
        def haversine_distance(lat1, lon1, lat2, lon2):
            p = math.pi / 180
            a = (
                    0.5 - math.cos((lat2 - lat1) * p) / 2 +
                    math.cos(lat1 * p) * math.cos(lat2 * p) * (1 - math.cos((lon2 - lon1) * p)) / 2
            )
            return 2 * 6371 * math.asin(math.sqrt(a))  # 6371 - Earth radius

        car_history = self.get_car_history(car_id, oauth_token=oauth_token, numdoc=max_skip + 1, until=timestamp)

        skipped = 0
        sessions = car_history['sessions']

        session_id = user_id = session_start = None

        for session in sessions:
            segment = session['segment']['diff']

            session_start = segment['start']['timestamp']

            if session_start > timestamp:
                continue

            mileage = segment['mileage']

            if mileage <= min_mileage:
                mileage_to_compare = haversine_distance(
                    segment['start']['latitude'], segment['start']['longitude'],
                    segment['finish']['latitude'], segment['finish']['longitude'],
                )
            else:
                mileage_to_compare = mileage

            if mileage_to_compare > min_mileage:
                session_id = session['segment']['session_id']
                user_id = session['user_details']['id']
                break

            skipped += 1

        if session_id is not None:
            nearest_session_info = {
                'session_id': session_id, 'user_id': user_id, 'session_start': session_start, 'skipped': skipped,
            }
        else:
            nearest_session_info = None

        return nearest_session_info

    def get_car_history(self, car_id, *, oauth_token, numdoc=None, since=None, until=None):
        return self._get_history(
            criteria={'car_id': car_id},
            oauth_token=oauth_token,
            numdoc=numdoc,
            since=since,
            until=until,
        )

    def get_user_history(self, user_id, *, oauth_token, numdoc, since=None, until=None):
        return self._get_history(
            criteria={'user_id': user_id},
            oauth_token=oauth_token,
            numdoc=numdoc,
            since=since,
            until=until
        )

    def _get_history(self, criteria, oauth_token, *, numdoc=None, since=None, until=None):
        url = self._urls.get_history_url(criteria=criteria, numdoc=numdoc, since=since, until=until)
        headers = {
        }
        try:
            return self._request(
                url=url, headers=headers, log=False, token=oauth_token,
            ).json()
        except requests.exceptions.HTTPError as e:
            if e.response.status_code == 404:
                return {
                    'sessions': []
                }
            raise

    @retry.retry(tries=3, delay=1)
    def _request(self, url, headers=None, token=None, log=True):
        h = self._default_headers.copy()
        h['Authorization'] = 'OAuth {}'.format(token)
        h['UserPermissionsCache'] = 'true'
        if headers:
            h.update(headers)

        response = self._session.request('GET', url, headers=h, timeout=self._timeout)
        if log:
            LOGGER.info('saas drive request: url=%s\tresponse=%s', url, response.text)
        response.raise_for_status()
        return response


class SaasDriveUrls:

    def __init__(self, root_url, service, prestable, version):
        self._root_furl = furl.furl(root_url)
        self._service = service
        self._prestable = prestable
        self._version = version
        self._arachnid_furl = furl.furl('http://arachnid09.search.yandex.net:12333')

    def _get_root_furl(self):
        return self._root_furl.copy()

    def _get_arachnid_furl(self):
        return self._arachnid_furl.copy()

    def get_fines_url(self):
        path = '/api/yandex/user/fines/list'
        args = self._build_args()
        url = self._get_root_furl().set(path=path, args=args).url
        return url

    def get_offer_info_url(self, oid):
        path = '/offer_info'
        args = self._build_args(oid=oid)
        url = self._get_root_furl().set(path=path, args=args).url
        return url

    def get_tracks_url(self, order_id):
        path = '/api/staff/track/trace'
        args = self._build_args(session=order_id)
        url = self._get_root_furl().set(path=path, args=args).url
        return url

    def get_nearest_session_url(self, violation_timestamp, car_id):
        path = '/api/staff/sessions/nearest'
        args = self._build_args(
            car_id=car_id,
            ride_dist=100,
            timestamp=violation_timestamp,
        )
        url = self._get_root_furl().set(path=path, args=args).url
        return url

    def get_history_url(self, criteria, numdoc, since, until):
        path = '/api/staff/sessions/history'

        limit_kwargs = self._filter_none_kwargs(numdoc=numdoc, since=since, until=until)
        criteria.update(limit_kwargs)

        args = self._build_args(**criteria)

        url = self._get_root_furl().set(path=path, args=args).url

        return url

    def _filter_none_kwargs(self, **kwargs):
        return {k: v for k, v in kwargs.items() if v is not None}

    def _build_args(self, **kwargs):
        kwargs['service'] = self._service
        if self._prestable:
            kwargs['prestable'] = 'true'
        if self._version:
            kwargs['version'] = self._version
        return kwargs


class SaasDriveStub:

    class OfferNotFoundError(Exception):
        pass

    def __init__(self):
        self._offers = {}

    def create_offer(self, oid, offer):
        self._offers[oid] = offer

    def build_offer_object(self, fix_price):
        return OfferInfo(
            vid=None,
            access_type=None,
            car_position=None,
            destination=None,
            distance_dest=None,
            fix_price=fix_price,
            time_dest=None,
        )

    def get_offer_info(self, oid, oauth_token):  # pylint: disable=unused-argument
        if oid not in self._offers:
            raise self.OfferNotFoundError
        return self._offers[oid]
