import contextlib
import furl
import json
import logging

from requests import HTTPError
from rest_framework.status import HTTP_204_NO_CONTENT

from django_yauth.util import get_real_ip
from kubiki.util import make_requests_session

import cars.settings

from cars.carsharing.models import DriveRoleAction, DriveRoleHierarchy
from cars.users.models import UserRole


LOGGER = logging.getLogger(__name__)


class SaasDriveAdminClient(object):
    DRIVE_USER_ID_REQUEST_FIELD = 'drive_user_id'
    DRIVE_USER_ACTIONS_REQUEST_FIELD = 'drive_user_actions'
    DRIVE_USER_ROLES_REQUEST_FIELD = 'drive_user_roles'

    def __init__(self, root_url, token, timeout=3, service='drive-frontend', version='v5.0.0', backend_cluster=None):
        self._urls = SaasDriveAdminUrls(root_url=root_url, service=service, version=version, backend_cluster=backend_cluster)

        self._token = token
        self._timeout = timeout

        self._session = make_requests_session()
        self._setup_session()

    def _setup_session(self):
        self._set_session_auth_token(self._token)
        self._session.verify = False

    def _set_session_auth_token(self, token):
        self._session.headers['Authorization'] = 'OAuth {}'.format(token)

    @classmethod
    def from_settings(cls, **kwargs_override):
        settings = cars.settings.SAAS_DRIVE

        kwargs = {
            'root_url': settings['url'],
            'token': settings['token'],
            'timeout': settings['timeout'],
            'service': settings['service'],
            'version': settings['version'],
            'backend_cluster': settings.get('backend_cluster', None),
        }
        kwargs.update(kwargs_override)

        return cls(**kwargs)

    @property
    def urls(self):
        return self._urls

    @contextlib.contextmanager
    def with_custom_auth(self, *, request=None, token=None, delegation_user_id=None):
        provided_auth_methods = [(request is not None), (token is not None), (delegation_user_id is not None)]

        assert provided_auth_methods.count(True) == 1, 'exactly one authentication method has to be provided'

        original_headers = self._session.headers.copy()
        original_cookies = self._session.cookies.copy()

        self._session.headers.pop('Authorization', None)
        self._session.headers.pop('HTTP_AUTHORIZATION', None)
        self._session.headers.pop('Cookie', None)
        self._session.headers.pop('HTTP_COOKIE', None)
        self._session.headers.pop('UserIdDelegation', None)

        if token is not None:
            self._set_session_auth_token(token)

        elif delegation_user_id is not None:
            self._set_session_auth_token(self._token)
            self._session.headers['UserIdDelegation'] = str(delegation_user_id)

        elif request is not None:
            if hasattr(request, '_request'):
                original_request = getattr(request, '_request')
            else:
                original_request = request

            if 'HTTP_AUTHORIZATION' in original_request.META:
                self._session.headers['Authorization'] = original_request.META['HTTP_AUTHORIZATION']

            if 'HTTP_COOKIE' in original_request.META:
                cookies = self._session.headers['Cookie'] = original_request.META['HTTP_COOKIE']

            user_ip = get_real_ip(original_request)
            if user_ip is not None:
                self._session.headers['X-Forwarded-For-Y'] = user_ip

        else:
            raise RuntimeError('no authentication method has been provided')

        try:
            yield self

        finally:
            self._session.cookies = original_cookies
            self._session.headers = original_headers

    @contextlib.contextmanager
    def with_robot_auth(self):
        with self.with_custom_auth(token=self._token):
            yield

    def _get_object_data(self, url, re_raise=True):
        data = None

        try:
            response = self._session.get(url, timeout=self._timeout)
            response.raise_for_status()
            data = response.json()
        except Exception:
            LOGGER.exception('unable to get object data')
            if re_raise:
                raise

        return data

    def _update_object_data(self, url, object_id, data, re_raise=True):
        response = None

        try:
            response = self._session.post(url, json=data, timeout=self._timeout)
            response.raise_for_status()
        except Exception:
            data_to_log = json.dumps(data)
            LOGGER.exception('unable to update object data: object - {}, data - {}'.format(object_id, data_to_log))
            if re_raise:
                raise

        try:
            data = response.json()
            tag_id = data.get('tagged_objects', [])[0].get('tag_id', [])[0]
            return tag_id
        except Exception:
            return None

    def add_user_tag(self, user_id, tag, comment='', *, re_raise=False, **extra_data):
        user_id = str(user_id)
        data = {'object_id': user_id, 'tag': tag, 'comment': comment}
        data.update(extra_data)
        return self._update_object_data(self._urls.get_user_tag_add_url(), user_id, data, re_raise=re_raise)

    def check_specific_user_tag(self, user_id, *tags, re_raise=False):
        user_tags = self.get_user_tags(user_id, re_raise=re_raise)
        specific_tag = next((t for t in user_tags if t['tag'] in tags), None)
        return specific_tag is not None

    def get_user_tags(self, user_id, *, re_raise=False):
        user_id = str(user_id)
        data = self._get_object_data(self._urls.get_user_tag_list_url(user_id), re_raise=re_raise)
        tags = data['records'] if data is not None else None
        return tags

    def add_user_problem_tag(self, user_id, tag, comment='', car_number='', session_id='', st_links=(), *, re_raise=False):
        st_links = (st_links, ) if isinstance(st_links, str) else st_links
        links = [{'uri': link, 'type': 'st'} for link in st_links]
        return self.add_user_tag(user_id, tag, comment, car_number=car_number, session_id=session_id, links=links, re_raise=re_raise)

    def update_email(self, user_id, email):
        data = {'id': str(user_id), 'email': email}
        self._update_object_data(self._urls.get_user_data_edit_url(), user_id, data)

    def update_phone(self, user_id, phone, is_phone_verified=True):
        data = {'id': str(user_id), 'phone': phone, "is_phone_verified": is_phone_verified}
        self._update_object_data(self._urls.get_user_data_edit_url(), user_id, data)

    def add_car_tag(self, car_id, tag_name, comment, priority=0, re_raise=False):
        car_id = str(car_id)
        url = self._urls.get_car_tag_add_url()
        data = {
            'car_id': car_id,
            'tag_name': tag_name,
            'comment': comment,
            'priority': priority,
        }
        self._update_object_data(url, car_id, data, re_raise)

    def remove_car_tag(self, tag_id, re_raise=False):
        tag_id = str(tag_id)
        url = self._urls.get_car_tag_remove_url(tag_id)
        self._update_object_data(url, tag_id, {}, re_raise)

    def get_car_tags(self, car_id, re_raise=False):
        car_id = str(car_id)
        data = self._get_object_data(self._urls.get_car_tag_list_url(car_id), re_raise=re_raise)
        tags = data['records'] if data is not None else None
        return tags

    def set_car_tag_performer(self, tag_id, re_raise=False):
        tag_id = str(tag_id)
        url = self._urls.get_car_tag_set_performer_url()
        data = {'tag_id': tag_id}
        self._update_object_data(url, tag_id, data, re_raise)

    def drop_car_tag_performer(self, *, tag_id_to_remove=None, tag_id_to_drop=None, re_raise=False):
        data = {}

        if tag_id_to_remove is not None:
            data['tag_id'] = str(tag_id_to_remove)

        if tag_id_to_drop is not None:
            data['drop_tag_ids'] = str(tag_id_to_drop)

        if not data:
            raise Exception('at least one tag id must be specified')

        _tag_id_to_report = ', '.join(data.values())

        url = self._urls.get_car_tag_drop_performer_url()
        self._update_object_data(url, _tag_id_to_report, data, re_raise)

    def evolve_car_tag(self, tag_id, tag_name, comment, priority=0, re_raise=False, **unused):
        tag_id = str(tag_id)
        url = self._urls.get_tag_evolve_url(tag_id)
        data = {
            'tag_name': tag_name,  # target tag name
            'comment': comment,
            'priority': priority,
        }
        self._update_object_data(url, tag_id, data, re_raise)

    def add_new_car_tags(self, car_id, comment):
        for tag_name in ('installation', 'new_car'):
            self.add_car_tag(
                car_id=car_id,
                tag_name=tag_name,
                comment=comment,
                priority=1000,
            )

    def check_access_to_deleting_user(self, user_id, request):
        has_access = True

        if user_id is not None:
            has_denied_role = self.check_user_role(
                *cars.settings.USERS['view_restrictions']['deleting']['denied_roles'],
                request=request,
                require_all=False
            )

            if has_denied_role:
                has_required_tags = self.check_specific_user_tag(
                    user_id, *cars.settings.USERS['view_restrictions']['deleting']['required_user_tags']
                )

                if has_required_tags:
                    has_access = False

        return has_access

    def check_user_role(self, *required_roles, user_id=None, request=None, require_all=False, use_proxy=True):
        if user_id is None and request is None:
            return False

        if not required_roles:
            return True

        if use_proxy:
            roles = set(self.iter_user_roles(user_id=user_id, request=request))
        else:
            if user_id is not None:
                roles = set(self.iter_user_db_roles(user_id))
            else:
                raise NotImplementedError('proxified role check is available by user id only')

        check = all if require_all else any
        has_role = check(required_role in roles for required_role in required_roles)

        return has_role

    def iter_user_roles(self, *, user_id=None, request=None):
        if request is not None:
            roles = self.get_request_user_roles(request)
        elif user_id is not None:
            roles = self.get_specific_user_roles(user_id)
        else:
            raise RuntimeError('either request or user id must be provided to check roles')

        for role in roles:
            try:
                is_active = bool(int(role['active']))
            except (KeyError, TypeError, ValueError):
                is_active = True

            if is_active:
                yield role['role_id']

    def get_specific_user_roles(self, user_id):
        roles = []

        try:
            response = self._session.get(self._urls.get_user_roles_list_url(user_id), timeout=self._timeout)
            response.raise_for_status()

            roles = response.json()['report'] or []

        except Exception:
            LOGGER.exception('unable to get user {} roles'.format(user_id))

        return roles

    def get_cached_request_user_roles(self, request, default=None):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ROLES_REQUEST_FIELD, host))
        return getattr(request, attr_name, default)

    def _set_cached_request_user_roles(self, request, roles):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ROLES_REQUEST_FIELD, host))
        setattr(request, attr_name, roles)

    def get_request_user_roles(self, request):
        cached_roles = self.get_cached_request_user_roles(request)

        if cached_roles is not None:
            return cached_roles

        roles = []

        try:
            # user may have no access to check his roles
            if request.user and request.user.id:
                user_id = request.user.id
                response = self._session.get(self._urls.get_user_roles_list_url(user_id), timeout=self._timeout)
            else:
                with self.with_custom_auth(request=request):
                    response = self._session.get(self._urls.get_user_roles_list_url(), timeout=self._timeout)

            response.raise_for_status()

            roles = response.json()['report'] or []

        except Exception:
            LOGGER.exception('unable to get user roles')

        # set roles anyway
        self._set_cached_request_user_roles(request, roles)

        return roles

    def iter_user_db_roles(self, user_id):
        if user_id is not None:
            yield from UserRole.objects.filter(user_id=str(user_id)).values_list('role_id', flat=True)

    def check_user_action(self, *required_actions, user_id=None, request=None, require_all=False, use_proxy=True):
        if user_id is None and request is None:
            return False

        if not required_actions:
            return True

        if use_proxy:
            actions = set(self.iter_user_actions(user_id=user_id, request=request))
        else:
            raise NotImplementedError('direct actions check is not currently supported')

        check = all if require_all else any
        has_action = check(required_action in actions for required_action in required_actions)

        return has_action

    def iter_user_actions(self, *, user_id=None, request=None, active_only=True):
        if request is not None:
            actions_list = self.get_request_user_actions(request)
        elif user_id is not None:
            actions_list = self.get_specific_user_actions(user_id)
        else:
            raise RuntimeError('either request or user id must be provided to check actions')

        yield from (action['action_id'] for action in actions_list if not active_only or action['enabled'])

    def get_specific_user_actions(self, user_id):
        actions_list = []

        try:
            response = self._session.get(self._urls.get_user_actions_list_url(user_id), timeout=self._timeout)
            response.raise_for_status()

            actions_list = response.json()['actions'] or []

        except Exception:
            LOGGER.exception('unable to get user {} actions'.format(user_id))

        return actions_list

    def get_cached_request_user_id(self, request, default=None):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ID_REQUEST_FIELD, host))
        return getattr(request, attr_name, default)

    def _set_cached_request_user_id(self, request, user_id):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ID_REQUEST_FIELD, host))
        setattr(request, attr_name, user_id)

    def get_cached_request_user_actions(self, request, default=None):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ACTIONS_REQUEST_FIELD, host))
        return getattr(request, attr_name, default)

    def _set_cached_request_user_actions(self, request, actions):
        host = self._urls.host.split('.', maxsplit=1)[0]
        attr_name = '_'.join((self.DRIVE_USER_ACTIONS_REQUEST_FIELD, host))
        setattr(request, attr_name, actions)

    def get_request_user_actions(self, request):
        cached_actions = self.get_cached_request_user_actions(request)

        if cached_actions is not None:
            return cached_actions

        user_id = None
        actions_list = []

        try:
            # user may have no access to check his actions
            if request.user and request.user.id:
                request_url = self._urls.get_user_actions_list_url(request.user.id)
                response = self._session.get(request_url, timeout=self._timeout)
            else:
                with self.with_custom_auth(request=request):
                    request_url = self._urls.get_user_actions_list_url()
                    response = self._session.get(request_url, timeout=self._timeout)

            response.raise_for_status()

            parsed_response = response.json()
            user_id = parsed_response['user_id']
            actions_list = parsed_response['actions'] or []

        except Exception:
            LOGGER.exception('unable to get actions')

        # set actions anyway
        self._set_cached_request_user_id(request, user_id)
        self._set_cached_request_user_actions(request, actions_list)

        return actions_list

    def get_raw_request_user_actions(self, request, *, re_raise=True):
        request_url = self._urls.get_user_actions_list_url()

        data = None
        user_id = None
        actions_list = []

        try:
            with self.with_custom_auth(request=request):
                response = self._session.get(request_url, timeout=self._timeout)

            response.raise_for_status()

            data = response.json()
            user_id = data['user_id']
            actions_list = data['actions'] or []

        except HTTPError as exc:
            LOGGER.error("unable to get actions: {}".format(exc.response.content))
            if re_raise:
                raise

        except Exception as exc:
            LOGGER.error("unable to get actions", exc_info=False)
            if re_raise:
                raise

        self._set_cached_request_user_id(request, user_id)
        self._set_cached_request_user_actions(request, actions_list)

        return data

    def get_all_roles_with_actions(self, actions):
        actions = (actions, ) if isinstance(actions, str) else actions

        target_roles = []

        role_ids_to_process = list(
            DriveRoleAction.objects
            .filter(action_id__in=actions)
            .values_list('role_id', flat=True)
        )

        while role_ids_to_process:
            target_roles.extend(role_ids_to_process)

            master_roles = (
                DriveRoleHierarchy.objects
                .filter(slave_role_id__in=role_ids_to_process)
                .values_list('role_id', flat=True)
            )

            role_ids_to_process = master_roles

        return target_roles

    def get_all_roles_list(self):
        all_role_ids = []

        try:
            response = self._session.get(self._urls.get_roles_list_url(), timeout=self._timeout)
            response.raise_for_status()

            all_roles = response.json()['report'] or []
            all_role_ids = [r['role_id'] for r in all_roles]

        except Exception:
            LOGGER.exception('unable to get roles list')

        return all_role_ids

    def get_all_users_with_roles_list(self, role, *, page=None, total_pages_limit=None):
        all_users = []

        try:
            request_url = self._urls.get_users_with_roles_list_url(role)

            if page is not None:
                request_url += '&page={}'.format(page)

            response = self._session.get(request_url, timeout=self._timeout)
            response.raise_for_status()

            empty_response = (response.status_code == HTTP_204_NO_CONTENT) and not response.content

            if not empty_response:
                response_data = response.json()

                curr_page_users = response_data['report']
                all_users.extend(curr_page_users)

                total_pages = response_data['pagination']['total_pages'] if page is None else 1

                if total_pages_limit is not None and total_pages > total_pages_limit:
                    raise Exception('total pages limit reached for role {}'.format(total_pages_limit))

                for extra_page in range(1, total_pages):
                    curr_page_users = self.get_all_users_with_roles_list(role, page=extra_page)
                    all_users.extend(curr_page_users)

        except Exception:
            LOGGER.exception('unable to get users with role %s', str(role))

        return all_users


class SaasDriveAdminUrls(object):
    def __init__(self, root_url, service, version, backend_cluster):
        self._root_furl = furl.furl(root_url)
        self._service = service
        self._version = version
        self._backend_cluster = backend_cluster

    @property
    def host(self):
        return self._root_furl.host

    def _get_root_furl(self):
        return self._root_furl.copy()

    def _build_args(self, **kwargs):
        kwargs['service'] = self._service
        if self._version:
            kwargs['version'] = self._version
        if self._backend_cluster:
            kwargs['backend_cluster'] = self._backend_cluster
        return kwargs

    def get_user_data_edit_url(self):
        path = '/api/staff/user/edit'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_car_tag_add_url(self):
        path = '/api/staff/car/tag/add'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_car_tag_list_url(self, car_id):
        path = '/api/staff/car/tag/list'
        url = self._get_root_furl().set(path=path, args=self._build_args(object_id=car_id)).url
        return url

    def get_car_tag_remove_url(self, tag_id):
        path = '/api/staff/car/tag/remove'
        url = self._get_root_furl().set(path=path, args=self._build_args(tag_id=tag_id)).url
        return url

    def get_car_tag_set_performer_url(self):
        path = '/api/staff/car/tag/perform/start'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_car_tag_drop_performer_url(self):
        path = '/api/staff/car/tag/perform/finish'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_tag_evolve_url(self, tag_id):
        path = '/api/staff/tag/evolve'
        url = self._get_root_furl().set(path=path, args=self._build_args(tag_id=tag_id)).url
        return url

    def get_user_tag_add_url(self):
        path = '/api/staff/user_tags/add'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_user_tag_list_url(self, user_id):
        path = '/api/staff/user_tags/list'
        url = self._get_root_furl().set(path=path, args=self._build_args(object_id=user_id)).url
        return url

    def get_roles_list_url(self):
        path = '/api/staff/roles/list'
        url = self._get_root_furl().set(path=path, args=self._build_args()).url
        return url

    def get_user_roles_list_url(self, user_id=None):
        path = '/api/staff/user/roles/list'
        extra_kwargs = {'user_id': str(user_id)} if user_id is not None else {}
        url = self._get_root_furl().set(path=path, args=self._build_args(**extra_kwargs)).url
        return url

    def get_user_actions_list_url(self, user_id=None):
        path = '/api/staff/user/actions/list'
        extra_kwargs = {'user_id': str(user_id)} if user_id is not None else {}
        url = self._get_root_furl().set(path=path, args=self._build_args(**extra_kwargs)).url
        return url

    def get_users_with_roles_list_url(self, roles):
        path = '/api/staff/roles/users'
        url = self._get_root_furl().set(path=path, args=self._build_args(roles=roles)).url
        return url
