import concurrent.futures
import datetime
import json
import logging
import operator
import time

import saaspy

import cars.settings
from cars.core.filter import CarFilter
from cars.core.models.car import Car
from cars.core.pedestrian_router import PedestrianRouter
from cars.core.util import euclidian_distance


LOGGER = logging.getLogger(__name__)


class SaasIndex(object):

    _converters = {}  # operator_id -> Converter mapping to transform SaaS doc into Car model.

    _prouter = PedestrianRouter()

    def __init__(self, saas_config, cache=None, delay=cars.settings.CAR_INDEX['DELAY']):
        self._client = saaspy.client.GeoKvSaasClient(**saas_config)
        self._cache = cache
        self._delay = delay

    @classmethod
    def register_converter(cls, converter_class):
        assert converter_class.operator not in cls._converters
        cls._converters[converter_class.operator] = converter_class

    def get_converter(self, op):
        converter_class = self._converters.get(op)
        if converter_class:
            return converter_class()
        return None

    def get_car(self, car_id):
        query = {
            'url': car_id,
        }
        query_string = self._client.construct_complex_query(query)

        response = self._client.search_by_text(query_string)
        response.raise_for_status()

        docs = response.documents
        assert len(docs) < 2
        if not docs:
            return None

        car = self._car_from_doc(docs[0])

        return car

    def get_cars(self, is_free=None, since=None):
        since_ts = None
        timestamp = int(time.time() - self._delay)
        if since:
            since_ts = int(since.timestamp())
            timestamp = max(timestamp, since_ts)

        query = {}
        if is_free is not None:
            query['is_free'] = 1 if is_free else 0
        if since_ts:
            query['updated_at'] = '>{}'.format(since_ts)
        if not query:
            query['url'] = '"*"'
        query_string = self._client.construct_complex_query(query)

        extra_params = {
            'numdoc': 10000,
            'balancertimeout': 1000,  # Allow heavy search requests.
            'snip': 'diqm=1',  # Disable snippet construction to speedup report.
            'relev': 'attr_limit=99999999',  # Enable more that 1000 responses.
        }

        response = self._client.search_by_text(query_string, extra_params=extra_params)
        response.raise_for_status()
        docs = response.documents

        res_cars = []
        for doc in docs:
            try:
                car = self._car_from_doc(doc)
            except Exception:
                LOGGER.exception('Failed to parse document:\n%s', doc)
                continue
            res_cars.append(car)

        timestamped_cars = TimestampedCars(timestamp, res_cars)

        return timestamped_cars

    def get_free_cars(self, car_filter=None, limit=None, rect=None):
        free_cars = None

        # Try to load from cache.
        if self._cache:
            free_cars = self._cache.get_free_cars()

        # Update cache if the entry has expired.
        if free_cars is None:
            free_cars = self.get_cars(is_free=True)
            if self._cache:
                self._cache.set_free_cars(free_cars)

        # Apply user filters.
        if car_filter:
            free_cars.cars = [car for car in free_cars.cars if car_filter.check(car)]

        # Filter by rectangle.
        if rect:
            sw, ne = rect  # pylint: disable=invalid-name
            free_cars.cars = [car for car in free_cars.cars
                              if sw[1] <= car.lat < ne[1] and sw[0] <= car.lon < ne[0]]

        # Pick any first N cars if the limit is provided.
        if limit:
            free_cars.cars = free_cars.cars[:limit]

        return free_cars

    def get_modified_cars(self, since, car_filter=None):
        modified_cars = self.get_cars(since=since)
        if car_filter:
            modified_cars.cars = [car for car in modified_cars.cars if car_filter.check(car)]
        return modified_cars

    def get_nearest_cars(self, center, car_filter=None, limit=None, rect=None):
        free_cars = self.get_free_cars(car_filter=car_filter, rect=rect)
        free_cars.cars = sorted(free_cars.cars, key=self._make_distance_sort_key(center))

        if limit:
            n_candidates = 2 * limit
            free_cars.cars = free_cars.cars[:n_candidates]

        self._enrich_cars_with_walk_time(free_cars, center)

        if limit:
            free_cars.cars = free_cars.cars[:limit]

        return free_cars

    def _make_distance_sort_key(self, center):
        def key(car):
            return euclidian_distance(center, (car.lon, car.lat))
        return key

    def _enrich_cars_with_walk_time(self, timestamped_cars, point):
        max_workers = max(len(timestamped_cars), 1)
        pool = concurrent.futures.ThreadPoolExecutor(max_workers=max_workers)
        futures = []

        for car in timestamped_cars.cars:
            future = pool.submit(self._set_car_walk_time, car, point)
            futures.append(future)

        for f in futures:
            f.result()
        pool.shutdown()

        timestamped_cars.cars = list(
            sorted(timestamped_cars.cars, key=operator.attrgetter('walk_time'))
        )

        return timestamped_cars

    def _set_car_walk_time(self, car, point):
        try:
            car.walk_time = self._prouter.get_walk_time((car.lon, car.lat), point)
        except Exception:
            LOGGER.exception('Failed to determine walk time')
            car.walk_time = 1800

    def _car_from_doc(self, doc):
        is_free = int(doc['is_free']) == 1
        op = doc['operator']

        city_id = doc.properties.get('city_id')
        if city_id is not None:
            city_id = int(city_id)

        updated_at = datetime.datetime.utcfromtimestamp(int(doc['updated_at']))
        data = json.loads(doc['data'])

        car = Car(
            id_=data['id'],
            local_id=data['local_id'],
            city_id=city_id,
            address_en=data.get('address_en', ''),
            address_ru=data.get('address_ru', ''),
            color=data['color'],
            fuel=data['fuel'],
            is_free=is_free,
            plate_number=data['plate_number'],
            lat=data['position']['lat'],
            lon=data['position']['lon'],
            model=data['model'],
            operator=op,
            discount=data.get('tariff', {}).get('discount', 0.0),
            parking_tariff=data.get('tariff', {}).get('parking', 0.0),
            usage_tariff=data.get('tariff', {}).get('usage', 0.0),
            transmission=data['transmission'],
            updated_at=updated_at,
        )
        return car


class TimestampedCars(object):

    def __init__(self, timestamp, source_cars):
        self.timestamp = timestamp
        self.cars = [car for car in source_cars if car.updated_at.timestamp() <= self.timestamp]

    def __len__(self):
        return len(self.cars)


class SaasIndexCache(object):

    _FREE_CARS_KEY = 'free_cars'

    def __init__(self, backend, timeout):
        self._backend = backend
        self.timeout = timeout

    def get_free_cars(self):
        return self._backend.get(self._FREE_CARS_KEY)

    def set_free_cars(self, free_cars):
        self._backend.set(self._FREE_CARS_KEY, free_cars, timeout=self.timeout)


class SaasIndexListener(object):

    def __init__(self, saas_config):
        self._saas_index = SaasIndex(saas_config)
        self._cars = {}  # Car.id -> Car
        self._updated_at = None
        self._on_car_updated_callbacks = []

    @property
    def cars(self):
        return list(self._cars.values())

    def setup(self):
        assert self._updated_at is None
        response = self._saas_index.get_cars()
        self._cars = {car.id: car for car in response.cars}
        self._updated_at = datetime.datetime.fromtimestamp(response.timestamp)
        LOGGER.info('SaaS index listener initialized with %s cars', len(self._cars))

    def listen(self):
        assert self._updated_at is not None, 'Car index is not initialized'

        car_filter = CarFilter(is_free=None)

        while True:
            response = self._saas_index.get_modified_cars(self._updated_at, car_filter=car_filter)

            for car in response.cars:
                old_car = self._cars.get(car.id)
                self._cars[car.id] = car

                for callback in self._on_car_updated_callbacks:
                    callback(old_car, car)

            self._updated_at = datetime.datetime.fromtimestamp(response.timestamp)

            time.sleep(1.5)

    def add_car_updated_callback(self, callback):
        self._on_car_updated_callbacks.append(callback)
