'''Module to use with Yandex.Sender: https://wiki.yandex-team.ru/users/lavrinenko/sender/'''

import enum
import requests
import simplejson

import furl

from cars.settings import SENDER as settings
from cars.core.util import import_class


class EmailNotifier:

    def __init__(self, sender):
        self._sender = sender

    @classmethod
    def from_settings(cls, *, sender_client_kwargs=None):
        sender_client_class = import_class(settings['client_class'])
        kwargs = {
            'url_template': settings['url_template'],
            'token': settings['token'],
        }
        if sender_client_kwargs:
            kwargs.update(sender_client_kwargs)
        sender = sender_client_class(**kwargs)
        return cls(sender)

    def send(self, *args, **kwargs):
        return self._sender.send(*args, **kwargs)


class SenderClient:

    def __init__(self, url_template, token, **kwargs):
        self._url_template = url_template
        self._token = token

    def send(self, *, campaign, to_email, args, attachments=None,
             async=False, for_testing=False, assert_ok=True):
        url = furl.furl(self._url_template.format(campaign=campaign))
        url.args = {
            'to_email': to_email
        }
        if for_testing:
            url.args['for_testing'] = 'true'
        headers = {
            'Authorization': 'Basic {}'.format(self._token)
        }
        data = {
            'async': str(async).lower(),
            'args': simplejson.dumps(args)
        }
        if attachments:
            data['attachments'] = simplejson.dumps(attachments)
        resp = requests.post(url.url, data=data, headers=headers, verify=False)
        resp.raise_for_status()
        data = resp.json()
        if assert_ok:
            assert data['result']['status'] == 'OK'
        return data


class SenderClientTesting(SenderClient):

    def send(self, **kwargs):
        kwargs['to_email'] = 'drive-fines@yandex-team.ru'
        return super().send(**kwargs)


class SenderClientCopyToMailing(SenderClient):

    def __init__(self, **kwargs):
        super().__init__(**kwargs)
        self.mailing = kwargs.pop('copy_to_mailing', 'drive-fines@yandex-team.ru')

    def send(self, **kwargs):
        result = super().send(**kwargs)
        kwargs['to_email'] = self.mailing
        super().send(**kwargs)
        return result
