from django.core.exceptions import ObjectDoesNotExist
from django.db import transaction

import cars.settings

from cars.carsharing.models.settings import DriveSettings, DriveSettingsHistory

from cars.core.history import HistoryManager


class SettingsManager(object):
    def __init__(self, default_operator_id):
        self._history_manager = HistoryManager(DriveSettingsHistory)
        self._default_operator_id = default_operator_id

    @classmethod
    def from_settings(cls):
        return cls(
            default_operator_id=cars.settings.COMMON['default_performer_id']
        )

    def get_value(self, key, *, default=None, use_ro_replica=True):
        qs = DriveSettings.objects.filter(setting_key=key)

        if use_ro_replica:
            qs.using(cars.settings.DB_RO_ID)

        instance = qs.first()

        value = instance.setting_value if instance else default
        return value

    def set_value(self, key, value, operator_id=None, *, only_if_differ=False):
        operator_id = operator_id or self._default_operator_id

        with transaction.atomic(savepoint=False):
            try:
                obj = DriveSettings.objects.select_for_update().get(setting_key=key)
            except ObjectDoesNotExist:
                obj = DriveSettings.objects.create(setting_key=key, setting_subkey='', setting_value=value)
                self._history_manager.add_entry(obj, operator_id)
            else:
                if not only_if_differ or obj.setting_value != value:
                    obj.setting_value = value
                    obj.save()

                    self._history_manager.update_entry(obj, operator_id)

        return value
