import collections
import copy
import unittest.mock

import solomon

import cars.settings

from .util import handle_error


def make_solomon_client(config=None):
    config = copy.deepcopy(config if config is not None else cars.settings.SOLOMON) or {}

    auth_provider = None
    token = config.pop('token', None)
    if token:
        auth_provider = solomon.OAuthProvider(token)

    if config:
        client = solomon.ThrottledPushApiReporter(auth_provider=auth_provider, **config)
    else:
        client = unittest.mock.MagicMock()

    return client


SOLOMON = make_solomon_client()


class SolomonHelper(object):
    def __init__(self, solomon_service, sensor_prefix):
        self._solomon = self._make_solomon_client(solomon_service)

        self._solomon_service = solomon_service
        self._sensor_prefix = sensor_prefix

    @property
    def solomon_service(self):
        return self._solomon_service

    @property
    def sensor_prefix(self):
        return self._sensor_prefix

    def _make_solomon_client(self, solomon_service):
        solomon_config = copy.deepcopy(cars.settings.SOLOMON)
        solomon_config['service'] = solomon_service
        client = make_solomon_client(solomon_config)
        return client

    def get_full_sensor_name(self, sensor_name):
        sensor_name = '.'.join((self.solomon_service, self.sensor_prefix, sensor_name))
        return sensor_name

    def report_value(self, sensor_name, value, *, re_raise=False):
        """
        :type value: int | float | list[int] | list[float]
        """
        with handle_error(re_raise=re_raise):
            full_sensor_name = self.get_full_sensor_name(sensor_name)
            report_name = [full_sensor_name] * len(value) if isinstance(value, collections.Sized) else full_sensor_name
            self._solomon.set_value(report_name, value)

    def increment_counter(self, sensor_name, increment=1, *, re_raise=False):
        if increment > 0:
            self.report_value(sensor_name, increment, re_raise=re_raise)

    def increment_counters(self, *, re_raise=False, **sensor_increment_mapping):
        for sensor_name, increment in sensor_increment_mapping.items():
            if increment > 0:
                self.report_value(sensor_name, increment, re_raise=re_raise)
