import itertools

import requests

from cars.settings import STARTREK as settings


class StartrekClient(object):
    def __init__(self, api_url, web_url, token):
        self._api_url = api_url
        self._web_url = web_url
        self._token = token

    @classmethod
    def from_settings(cls):
        return cls(
            api_url=settings['api_url'],
            web_url=settings['web_url'],
            token=settings['token'],
        )

    @property
    def api_url(self):
        return self._api_url

    @property
    def web_url(self):
        return self._web_url

    def _perform_request(self, method, handlers, **kwargs):
        url = '/'.join(itertools.chain((self._api_url, 'v2'), handlers))
        self._add_auth(kwargs)
        response = requests.request(method, url, **kwargs)
        response.raise_for_status()
        return response.json()

    def _add_auth(self, dct):
        dct.setdefault('headers', {}).setdefault(
            'Authorization', 'OAuth {}'.format(self._token)
        )

    def search_issue(self, *, query, **extra):
        data = {
            'query': str(query),
        }

        data.update(extra)

        response_data = self._perform_request(method='post', handlers=['issues', '_search'], json=data)
        return response_data

    def create_issue(self, *, queue, summary, description, followers=None, **extra):
        data = {
            'queue': queue,
            'summary': summary,
            'description': description,
        }

        if followers:
            data['followers'] = followers

        data.update(extra)

        response_data = self._perform_request(method='post', handlers=['issues'], json=data)
        return response_data

    def get_issue(self, issue_key):
        response_data = self._perform_request(method='get', handlers=['issues', issue_key])
        return response_data

    def patch_issue(self, issue_key, *, summary=None, description=None, followers=None):
        data = {
            'summary': summary,
            'description': description,
            'followers': followers,
        }

        data = {k: v for k, v in data.items() if v is not None}

        response_data = self._perform_request(method='patch', handlers=['issues', issue_key], json=data)
        return response_data

    def get_issue_comments(self, issue_key):
        # return only 50 comments by default
        response_data = self._perform_request(method='get', handlers=['issues', issue_key, 'comments'])
        return response_data

    def add_issue_comment(self, issue_key, *, text='', summonees=(), attachment_ids=()):
        summonees = [summonees] if not isinstance(summonees, (tuple, list)) else list(summonees)
        attachment_ids = [str(id_) for id_ in attachment_ids]
        data = {
            'text': text,
            'summonees': summonees,
            'attachmentIds': attachment_ids,
            # 'isAddToFollowers': add_to_followers,  # incorrect data format if boolean is used
        }
        response_data = self._perform_request(method='post', handlers=['issues', issue_key, 'comments'], json=data)
        return response_data

    def upload_attachment(self, file_path, issue_key=None, file_name=None):
        handlers = ['issues', issue_key, 'attachments'] if issue_key is not None else ['attachments']
        params = {'filename': file_name} if file_name is not None else {}
        files = {'file': open(file_path, 'rb')}
        response_data = self._perform_request(method='post', handlers=handlers, params=params, files=files)
        return response_data
