from collections import namedtuple, Mapping
import enum
import gevent
import hashlib
from itertools import chain
import json
import logging
import requests
import socket
from urllib.parse import urlencode

from kubiki.util import make_requests_session

from cars.core.util import datetime_helper, phone_number_helper
import cars.settings

LOGGER = logging.getLogger(__name__)


class BadTelephonyApiResponseError(Exception):
    def __init__(self, *args, error_status=None, error_message=''):
        super().__init__(*args)
        self.__error_status = error_status
        self.__error_message = error_message

    @property
    def error_status(self):
        return self.__error_status

    @property
    def error_message(self):
        return self.__error_message

    def __str__(self):
        return (
            'BadTelephonyApiResponseError: status - {}, message - {}'
            .format(self.__error_status, self.__error_message)
        )


class TelephonyQueue(enum.Enum):
    CARSHARING = 'carsharing'
    CARSHARING_SPB = 'carsharing-spb'
    CARSHARING_KAZAN = 'carsharing-kzn'
    CARSHARING_SOCHI = 'carsharing-sochi'
    CARSHARING_VIP = 'carsharing-vip'
    CARSHARING_ENG = 'carsharing-eng'
    CARSHARING_CORP = 'carsharing-corp'


DYNAMIC_TELEPHONY_QUEUE = 'carsharing_dynamic'


ORDINARY_TELEPHONY_QUEUES = (
    TelephonyQueue.CARSHARING,
    TelephonyQueue.CARSHARING_SPB,
    TelephonyQueue.CARSHARING_KAZAN,
    # TelephonyQueue.CARSHARING_SOCHI,  # check permissions
)


class TelephonyCallCenter(enum.Enum):
    AUDIOTELE = 'ATELE'
    BEEPER = 'BEEPER'
    NEXTCONTACT = 'NEXTCONTACT'
    YANDEX = 'YNDXCC'


class TelephonyHelperBase(object):
    def __init__(self, request_timeout=15, retries=3, pool_connections=10, pool_maxsize=10):
        self._session = make_requests_session(
            retries=retries, pool_connections=pool_connections, pool_maxsize=pool_maxsize
        )
        self._request_timeout = request_timeout

    def _setup_authorization_headers(self, api_token):
        self._session.headers.update({'Authorization': 'OAuth {}'.format(api_token)})

    def _perform_request(self, url, *, raise_for_status=False, exc_info=True, **kwargs):
        LOGGER.info('telephony request: %s', url)

        data = None

        try:
            method = kwargs.pop('method', 'get')
            timeout = kwargs.pop('timeout', self._request_timeout)
            response = self._session.request(method, url, timeout=timeout, **kwargs)
            response.raise_for_status()
            data = response.json()
        except Exception as exc:
            status = None
            text = ''

            if isinstance(exc, requests.HTTPError):
                if exc.response is not None:
                    status = exc.response.status_code
                    text = exc.response.text

            text = text.strip() or str(exc)

            LOGGER.exception(
                'exception processing telephony request: status - {}, content - {}.'.format(status, text),
                exc_info=exc_info
            )

            if raise_for_status:
                raise

        return data


class TelephonyApiHelper(TelephonyHelperBase):
    def __init__(self, base_url, api_url, user_name, user_token, request_timeout=15, retries=3):
        super().__init__(request_timeout, retries)

        self._base_url = base_url
        self._api_url = api_url

        self._api_user_name = user_name
        self._api_user_token = user_token

        self._host_fqdn = socket.getfqdn()  # consider to use ip address if you are using a local machine

    @classmethod
    def _from_settings(cls, api_url):
        call_center_settings = cars.settings.CALLCENTER

        base_url = call_center_settings['base_url']
        user_name = call_center_settings['api']['user']
        user_token = call_center_settings['api']['secret_key']
        request_timeout = call_center_settings['request_timeout']

        return cls(base_url, api_url, user_name, user_token, request_timeout)

    def _get_authorized_request_url(self, *, api_url, handlers, params):
        params.setdefault('dauth', self._api_user_name)
        params.setdefault('dhost', self._host_fqdn)
        params.setdefault('dtime', datetime_helper.timestamp_now(truncate=True))

        api_url = '/'.join(chain((api_url, ), handlers))

        request_params_url = '&'.join((api_url, urlencode(params)))

        digest = hashlib.sha1((request_params_url + self._api_user_token).encode('ascii')).hexdigest()

        authorized_request_url = '{}{}&dsign={}'.format(self._base_url, request_params_url, digest)
        return authorized_request_url

    def _perform_request(self, url, *, raise_for_status=False, ok_status=(200, ), **kwargs):
        data = None

        method = kwargs.get('method', 'get')

        raw_data = super()._perform_request(url, raise_for_status=raise_for_status, **kwargs)

        if raw_data is not None:
            status = raw_data.get('STATUSCODE')
            text = raw_data.get('STATUSMSG')

            if status in ok_status:
                if method == 'get':
                    data = raw_data['DATA'] or {}

            else:
                error_message = 'Bad telephony response (error code {}): {}'.format(status, text)
                LOGGER.error(error_message)

                if raise_for_status:
                    raise BadTelephonyApiResponseError(error_status=status, error_message=text)

        return data


class TimeRangeTelephonyApiHelper(TelephonyApiHelper):
    def get_time_range_data(self, start_timestamp, end_timestamp):
        request_handlers = self._get_api_handlers()
        request_params = self._get_api_params(start_timestamp, end_timestamp)

        authorized_request_url = self._get_authorized_request_url(
            api_url=self._api_url, handlers=request_handlers, params=request_params
        )

        data = self._perform_request(authorized_request_url)

        if data is not None:
            data = [self._process_entry(entry) for entry in data.values()]

        return data

    def _get_api_handlers(self):
        return ()

    def _get_api_params(self, start_timestamp, end_timestamp):
        raise NotImplementedError

    def _process_entry(self, entry):
        return entry


class IncomingTelephonyApiHelper(TimeRangeTelephonyApiHelper):
    API_RESPONSE = namedtuple(
        'API_RESPONSE', ['time_id', 'queue', 'call_id', 'agent', 'verb', 'data', ]
    )

    @classmethod
    def from_settings(cls):
        api_url = cars.settings.CALLCENTER['api']['url']
        return cls._from_settings(api_url)

    def _get_api_params(self, start_timestamp, end_timestamp):
        return self._get_in_calls_api_params(start_timestamp, end_timestamp, [q.value for q in TelephonyQueue])

    def _get_in_calls_api_params(self, start_timestamp, end_timestamp, queue_names):
        request_template = {
            'SECTION': 'QM',
            'TYPE': 'DBLIST_NOTI',
            'QUEUEDB': 'direct',
            'QUEUELIST': queue_names,
            'TIMERANGE': {
                'START': '{}'.format(start_timestamp),
                'END': '{}'.format(end_timestamp),
            },
        }

        params = {'request': json.dumps(request_template, separators=(',', ':'))}
        return params

    def _process_entry(self, entry):
        return self.API_RESPONSE(
            time_id=datetime_helper.timestamp_to_datetime(entry['time_id']),
            queue=entry['queue'],
            call_id=entry['call_id'],
            agent=entry['agent'],
            verb=entry['verb'],
            data=entry['data2'],
        )


class OutgoingTelephonyApiHelper(TimeRangeTelephonyApiHelper):
    API_RESPONSE = namedtuple(
        'API_RESPONSE', ['time_enter', 'time_connect', 'time_exit', 'agent', 'phone', 'duration', ]
    )

    def __init__(self, base_url, api_url, user_name, user_token, request_timeout=15, retries=3):
        super().__init__(base_url, api_url, user_name, user_token, request_timeout, retries)
        self._internal_phone_numbers_to_collect = self._get_internal_phone_numbers_to_collect()

    def _get_internal_phone_numbers_to_collect(self):
        work_phones = []
        return work_phones

    @classmethod
    def from_settings(cls):
        api_url = cars.settings.CALLCENTER['api']['stat_url']
        return cls._from_settings(api_url)

    def _get_api_params(self, start_timestamp, end_timestamp):
        return self._get_out_calls_api_params(start_timestamp, end_timestamp, self._internal_phone_numbers_to_collect)

    def _get_out_calls_api_params(self, start_timestamp, end_timestamp, internal_phone_numbers):
        request_template = {
            'SECTION': 'INTERNAL',
            'DIRECTION': 'O',
            'TIMERANGE': {
                'START': '{}'.format(start_timestamp),
                'END': '{}'.format(end_timestamp)
            },
            'TYPE': 'LIST',
            'NUMLIST': {
                'TYPE': 'LIST',
                'DATA': internal_phone_numbers,
            },
        }

        params = {'request': json.dumps(request_template, separators=(',', ':'))}
        return params

    def get_time_range_data(self, start_timestamp, end_timestamp):
        data = super().get_time_range_data(start_timestamp, end_timestamp)
        if data is not None:
            data = [entry for entry in data if entry.phone]
        return data

    def _process_entry(self, entry):
        phone = phone_number_helper.normalize_phone_number(entry['CALLEDFINAL'])

        if int(entry['TIMECONNECT']):  # time connect may be '0' or timestamp
            time_connect = datetime_helper.timestamp_to_datetime(entry['TIMECONNECT'])
        else:
            time_connect = None

        return self.API_RESPONSE(
            time_enter=datetime_helper.timestamp_to_datetime(entry['TIMEORIG']),
            time_connect=time_connect,
            time_exit=datetime_helper.timestamp_to_datetime(entry['TIMEDISCONNECT']),
            phone=phone,
            duration=int(entry['DURATION']),
            agent=entry['CALLING'],
        )


class SourceRoutingTelephonyApiHelper(TelephonyApiHelper):
    REDIRECT_ACTION_ID = '10'
    ALL_REDIRECT_ACTION_IDS = ('1', '10')
    TERMINATE_CALL_ACTION_ID = '400'

    @classmethod
    def from_settings(cls):
        api_url = cars.settings.CALLCENTER['api']['source_routing_url']
        return super()._from_settings(api_url)

    def _get_url(self, *, queue_name, handlers, params=None):
        return self._get_authorized_request_url(
            api_url=self._api_url, handlers=[queue_name] + handlers, params=params or {}
        )

    def _perform_multiple_requests(self, urls):
        return gevent.joinall([
            gevent.spawn(self._perform_request, url)
            for url in urls
        ])

    def _process_phone_number(self, phone_number):
        return phone_number.lstrip('+')  # "plus" sign is not correctly processed by telephony

    def _process_phone_number_back(self, phone_number):
        # telephony can store phone numbers without "plus" sign
        return '+{}'.format(phone_number) if not phone_number.startswith('+') else phone_number

    def add_to_queue(self, entries, queue_names):
        queue_names = queue_names if isinstance(queue_names, (tuple, list)) else (queue_names, )
        urls = (
            self._get_url(
                queue_name=queue_name,
                handlers=['add'],
                params={
                    '_data': json.dumps(entry)
                },
            )
            for entry in entries
            for queue_name in queue_names
        )
        self._perform_multiple_requests(urls)

    def remove_from_queue(self, phones, queue_names):
        queue_names = queue_names if isinstance(queue_names, (tuple, list)) else (queue_names, )
        urls = (
            self._get_url(
                queue_name=queue_name,
                handlers=['delete', phone])
            for phone in phones
            for queue_name in queue_names
        )
        self._perform_multiple_requests(urls)

    def iter_queue_phones(self, actions, queue_names):
        actions = actions if isinstance(actions, (list, tuple)) else (actions, )
        queue_names = queue_names if isinstance(queue_names, (tuple, list)) else (queue_names, )

        data_mappings = [
            self._perform_request(self._get_url(
                queue_name=queue_name,
                handlers=['list'],
            ))
            for queue_name in queue_names
        ]

        for data_mapping in data_mappings:
            for x in (data_mapping or {}).values():  # data_mapping can be None if error
                if actions is None or x['ACTIONID'] in actions:
                    yield x['SRCNUMB']

    def add_call_priority(self, *, phones):
        phones = [self._process_phone_number(p) for p in phones]
        entries = (
            {
                "SRCNUMB": phone,
                "ACTIONID": self.REDIRECT_ACTION_ID,
                "ACTIONDATA": TelephonyQueue.CARSHARING_VIP.value
            }
            for phone in phones
        )
        self.add_to_queue(entries, TelephonyQueue.CARSHARING.value)

    def delete_call_priority(self, *, phones):
        phones = [self._process_phone_number(p) for p in phones]
        self.remove_from_queue(phones, TelephonyQueue.CARSHARING.value)

    def list_call_priority_phones(self):
        phones = [
            self._process_phone_number_back(p)
            for p in self.iter_queue_phones(self.ALL_REDIRECT_ACTION_IDS, TelephonyQueue.CARSHARING.value)
        ]
        return phones

    def add_to_blacklist(self, *, phones):
        queue_names = [x.value for x in ORDINARY_TELEPHONY_QUEUES]
        phones = [self._process_phone_number(p) for p in phones]

        for queue_name in queue_names:
            entries = (
                {
                    "SRCNUMB": phone,
                    "ACTIONID": self.TERMINATE_CALL_ACTION_ID,
                    "ACTIONDATA": queue_name,
                } for phone in phones
            )
            self.add_to_queue(entries, queue_name)

    def remove_from_blacklist(self, *, phones):
        queue_names = [x.value for x in ORDINARY_TELEPHONY_QUEUES]
        phones = [self._process_phone_number(p) for p in phones]
        self.remove_from_queue(phones, queue_names)

    def list_blacklisted_phones(self):
        queue_names = [x.value for x in ORDINARY_TELEPHONY_QUEUES]
        phones = [
            self._process_phone_number_back(p)
            for p in self.iter_queue_phones(self.TERMINATE_CALL_ACTION_ID, queue_names)
        ]
        return phones


class TelephonySettingsHelper(TelephonyApiHelper):
    @classmethod
    def from_settings(cls):
        call_center_settings = cars.settings.CALLCENTER
        api_url = {
            'get': call_center_settings['api']['get_settings_url'],
            'set': call_center_settings['api']['set_settings_url'],
        }
        return cls._from_settings(api_url)

    def load_settings(self, application_source):
        api_url = self._api_url['get'].format(application_source=application_source)
        url = self._get_authorized_request_url(api_url=api_url, handlers=(), params={})
        data = self._perform_request(url, raise_for_status=True)
        content = data.get('CONTENT', {}) if data else {}
        value_mapping = {key: value['VALUE'] for key, value in content.items()}
        return value_mapping

    def set_settings(self, value_mapping, application_source):
        settings_value = json.dumps(value_mapping, separators=(',', ':'))
        api_url = self._api_url['set'].format(application_source=application_source)
        url = self._get_authorized_request_url(
            api_url=api_url, handlers=('DISTRIB', ), params={'_DATA': settings_value}
        )
        self._perform_request(url, method='post', ok_status=(200, 304), raise_for_status=True)
