'''Module to use with Toloka'''
import logging
import random
import retry
import time

from django.utils import timezone

from cars.ext.toloka import api
from cars.settings import TOLOKA as settings
from cars.core.util import import_class


LOGGER = logging.getLogger(__name__)


class TolokaProjectManager:

    CLONE_POOL_TIMEOUT = 30

    def __init__(self, *, toloka_client, project_id, template_pool_id=None):
        self._client = toloka_client
        self._project_id = project_id
        self._template_pool_id = template_pool_id

    @classmethod
    def from_settings(cls, project_id, template_pool_id=None):
        toloka_client_class = import_class(settings['client_class'])
        toloka_client = toloka_client_class(
            oauth_token=settings['token'],
            uri=settings['url'],
        )
        return cls(
            toloka_client=toloka_client,
            project_id=project_id,
            template_pool_id=template_pool_id,
        )

    def get_current_pool_id(self, *, modifier=None):
        private_name = timezone.now().strftime('%Y-%m-%d')
        if modifier:
            private_name += '_{}'.format(modifier)

        existing_private_names = {
            p['private_name']: p['id']
            for p in self._client.pools(project_id=self._project_id)
        }
        if private_name in existing_private_names:
            return existing_private_names[private_name]

        clone_response = self._client.pools(self._template_pool_id).clone()
        operation_id = clone_response['id']

        clone_pool_start_time = time.time()
        while (clone_response['status'] in ('PENDING', 'RUNNING')
               and time.time() - clone_pool_start_time < self.CLONE_POOL_TIMEOUT):
            time.sleep(1)
            clone_response = self._client.operations(operation_id).get()

        if clone_response['status'] != 'SUCCESS':
            raise RuntimeError(
                'failed to clone pool, clone response: {}'.format(clone_response)
            )

        pool_id = int(clone_response['details']['pool_id'])
        pool_properties = self._client.pools(pool_id).get()
        pool_properties['private_name'] = private_name
        self._client.pools(pool_id).update(pool_properties)
        return pool_id

    @staticmethod
    def _get_permutation_straight_and_reversed(n):
        permutation = list(range(n))
        random.shuffle(permutation)
        reverse_permutation = [0 for _ in range(n)]
        for i in range(len(permutation)):
            reverse_permutation[permutation[i]] = i
        return (permutation, reverse_permutation)

    def start_task_suite(self, *, pool_id, tasks):
        '''Creates a pool, adds the tasks and starts the pool.'''

        permutation, reverse_permutation = self._get_permutation_straight_and_reversed(len(tasks))

        permutted_tasks = [tasks[i] for i in permutation]
        task_suite_response = self._client.task_suites(
            allow_defaults=True,  # overlap from pool, not from task
            open_pool=True,
        ).create([{
            'pool_id': pool_id,
            'tasks': permutted_tasks,
        }])[0]
        task_results = task_suite_response['tasks']
        task_suite_response['tasks'] = [task_results[i] for i in reverse_permutation]

        return task_suite_response

    def get_users_finished_task_suite_count(self, task_suite_id):
        return len(set(
            a['id'] for a in self._client.assignments(
                task_suite_id=task_suite_id,
                status='ACCEPTED'
            )
        ))

    @retry.retry(tries=3, delay=1)
    def get_task_suite_result(self, task_suite_id):
        result = {}

        for user_data in self._client.assignments(
            task_suite_id=task_suite_id,
        ):
            if 'solutions' not in user_data:
                continue
            for task, solution in zip(user_data['tasks'], user_data['solutions']):
                dct = result.setdefault(task['id'], {})
                for k, v in solution['output_values'].items():
                    dct.setdefault(k, []).append(v)
        return result

    def find_control_fails(self, task_suite_id):
        control_fails = []
        for user_data in self._client.assignments(
            task_suite_id=task_suite_id,
        ):
            if 'solutions' not in user_data:
                continue
            for task, solution in zip(user_data['tasks'], user_data['solutions']):
                if not 'known_solutions' in task:
                    continue
                if not solution['output_values'] in [
                        ks['output_values'] for ks in task['known_solutions']
                ]:
                    control_fails.append((task, solution))
        return control_fails
