import collections
import copy
import decimal
import enum
import random
import time
import logging
import uuid

import furl
import kubiki.util
import requests

import cars.settings


LOGGER = logging.getLogger(__name__)


class TrustClient:

    class Error(Exception):
        pass

    class BadRequestError(Error):
        pass

    class InvalidPaymentMethodError(Error):
        pass

    def __init__(self, root_url, service_token, default_product_id,
                 timeout, verify_ssl, push_client, back_url=None):
        self._service_token = service_token
        self._default_product_id = default_product_id
        self._timeout = timeout
        self._verify_ssl = verify_ssl
        self._urls = TrustUrls(root_url=root_url, back_url=back_url)
        self._session = kubiki.util.make_requests_session()
        self._push_client = push_client

    @classmethod
    def from_settings(cls, push_client):
        trust_settings = cars.settings.TRUST
        return cls(
            root_url=trust_settings['root_url'],
            service_token=trust_settings['service_token'],
            default_product_id=trust_settings['default_product_id'],
            timeout=trust_settings['timeout'],
            verify_ssl=trust_settings['verify_ssl'],
            push_client=push_client,
            back_url=trust_settings['back_url'],
        )

    def get_payment_methods(self, uid):
        url = self._urls.get_payment_methods()
        headers = self._get_default_headers(uid=uid)
        response = self._request(url=url, headers=headers)
        return response.json()

    def get_payment(self, uid, purchase_token):
        url = self._urls.get_payment(purchase_token)
        headers = self._get_default_headers(uid=uid)
        response = self._request(url=url, headers=headers)
        return response.json()

    def create_payment(self, uid, paymethod_id, amount, user_email,
                       currency='RUB', fiscal_title=None, fiscal_nds=None):
        url = self._urls.create_payment()
        headers = self._get_default_headers(uid=uid)
        data = {
            'paymethod_id': paymethod_id,
            'amount': amount,
            'currency': currency,
            'product_id': self._default_product_id,
            'user_email': user_email,
        }
        if self._urls.get_back_url():
            data['back_url'] = self._urls.get_back_url()
        if fiscal_title:
            data['fiscal_title'] = fiscal_title
        if fiscal_nds:
            data['fiscal_nds'] = fiscal_nds

        try:
            response = self._request(method='POST', url=url, headers=headers, data=data)
        except requests.HTTPError as e:
            LOGGER.exception('http error in trust request for uid = %s', uid)
            if e.response is not None and e.response.status_code == 400:
                response_data = e.response.json()
                LOGGER.error('trust response_data for uid %s: %s', uid, response_data)
                if response_data.get('status_code') == 'invalid_payment_method':
                    raise self.InvalidPaymentMethodError
                else:
                    LOGGER.warning('unhandled bad request for uid = %s: %s',
                                   uid, response_data)
            raise
        except Exception:
            LOGGER.exception('failed to ask trust for uid = %s', uid)
            raise

        return response.json()

    def start_payment(self, uid, purchase_token):
        return self._request_payment_action(
            uid=uid,
            purchase_token=purchase_token,
            action='start',
        )

    def clear_payment(self, uid, purchase_token):
        return self._request_payment_action(
            uid=uid,
            purchase_token=purchase_token,
            action='clear',
        )

    def unhold_payment(self, uid, purchase_token):
        return self._request_payment_action(
            uid=uid,
            purchase_token=purchase_token,
            action='unhold',
        )

    def refund_payment(self, uid, purchase_token):
        payment_response = self.get_payment(uid=uid, purchase_token=purchase_token)

        refund_payload = {
            'purchase_token': purchase_token,
            'reason_desc': 'Cancel payment',
            'orders': [
                {
                    'order_id': order['order_id'],
                    'delta_amount': order['paid_amount'],
                }
                for order in payment_response['orders']
                if decimal.Decimal(order['paid_amount']) > 0
            ],
        }
        if not refund_payload['orders']:
            LOGGER.info('nothing to refund for payment %s', purchase_token)
            return

        create_refund_response = self._request(
            method='POST',
            url=self._urls.create_refund(),
            headers=self._get_default_headers(uid=uid),
            data=refund_payload,
        )
        create_refund_response_data = create_refund_response.json()
        trust_refund_id = create_refund_response_data['trust_refund_id']

        self._request(
            method='POST',
            url=self._urls.start_refund(trust_refund_id=trust_refund_id),
            headers=self._get_default_headers(uid=uid),
            expected_status='wait_for_notification',
        )

    def _request_payment_action(self, uid, purchase_token, action):
        url = self._urls.payment_action(purchase_token=purchase_token, action=action)
        headers = self._get_default_headers(uid=uid)
        response = self._request(method='POST', url=url, headers=headers)

        response_data = response.json()
        if response_data['status'] == 'error':
            raise self.Error(response_data['status_code'])

        return response_data

    def resize_payment(self, uid, purchase_token, amount, qty):
        payment = self.get_payment(uid=uid, purchase_token=purchase_token)
        orders = [order for order in payment['orders']]
        assert len(orders) == 1, 'can resize only single order payments'
        order_id = orders[0]['order_id']

        url = self._urls.resize_payment_order_url(purchase_token=purchase_token, order_id=order_id)
        headers = self._get_default_headers(uid=uid)
        data = {
            'amount': amount,
            'qty': qty,
            'data': None,  # Missing this parameter results in `unknown_error` from Trust.
        }
        response = self._request(url, method='POST', headers=headers, data=data)

        response_data = response.json()
        if response_data['status'] == 'error':
            raise self.Error(response_data['status_code'])

        return response_data

    def _get_default_headers(self, uid):
        return {
            'X-Service-Token': self._service_token,
            'X-Uid': str(uid),
        }

    def _request(self, url, method='GET', headers=None, data=None, expected_status='success'):
        response = self._session.request(
            method=method,
            url=url,
            headers=headers,
            json=data,
            timeout=self._timeout,
            verify=self._verify_ssl,
        )

        try:
            response.raise_for_status()
        except Exception as e:
            self._log_request(
                method=method,
                url=url,
                headers=headers,
                data=data,
                response=response,
            )
            # if (isinstance(e, requests.HTTPError)
            #         and e.response is not None  # pylint: disable=no-member
            #         and e.response.status_code == 400):  # pylint: disable=no-member
            #     raise self.BadRequestError
            # else:
            raise

        self._log_request(method=method, url=url, headers=headers, data=data, response=response)

        response_data = response.json()
        if response_data['status'] != expected_status:
            raise self.Error(response_data)

        return response

    def _log_request(self, method, url, headers, data, response):
        if self._push_client is None:
            LOGGER.warning('logging disabled')
            return

        data = {
            'request': {
                'method': method,
                'url': url,
                'headers': self._prepare_headers_for_logging(headers),
                'data': self._prepare_data_for_logging(data),
            },
            'response': None,
        }
        if response is not None:
            data['response'] = {
                'status_code': response.status_code,
                'data': response.json(),
            }

        try:
            self._push_client.log(
                type_='trust.reqans',
                data=data,
            )
        except Exception:
            LOGGER.exception('failed to log trust payments request')

    def _prepare_headers_for_logging(self, headers):
        if not headers:
            return headers

        headers = copy.deepcopy(headers)
        if 'X-Service-Token' in headers:
            headers['X-Service-Token'] = '*' * len(headers['X-Service-Token'])

        return headers

    def _prepare_data_for_logging(self, data):
        if data is None:
            return None

        data = copy.deepcopy(data)
        for key, value in data.items():
            if isinstance(value, decimal.Decimal):
                data[key] = float(value)

        return data


class TrustUrls:

    def __init__(self, root_url, back_url):
        self._root_furl = furl.furl(root_url)
        self._back_url = back_url

    def _get_base_furl(self):
        return self._root_furl.copy().set(path='/trust-payments/v2')

    def get_back_url(self):
        return self._back_url

    def get_payment_methods(self):
        return self._get_base_furl().add(path='/payment-methods').url

    def get_payment(self, purchase_token):
        return self._get_base_furl().add(path='/payments/{}'.format(purchase_token)).url

    def payment_action(self, purchase_token, action):
        return (
            self._get_base_furl()
            .add(path='/payments/{}/{}'.format(purchase_token, action))
            .url
        )

    def create_payment(self):
        return self._get_base_furl().add(path='/payments').url

    def resize_payment_order_url(self, purchase_token, order_id):
        path = '/payments/{}/orders/{}/resize'.format(purchase_token, order_id)
        return self._get_base_furl().add(path=path).url

    def create_refund(self):
        return self._get_base_furl().add(path='/refunds').url

    def start_refund(self, trust_refund_id):
        path = '/refunds/{}/start'.format(trust_refund_id)
        return self._get_base_furl().add(path=path).url


class StubTrustClient:
    # pylint: disable=unused-argument

    _payment_methods = collections.defaultdict(dict)
    _payments = collections.defaultdict(dict)
    _products = {}

    class PostStartAction(enum.Enum):
        DO_NOTHING = 0
        AUTHORIZE = 1
        UNAUTHORIZE_NO_REASON = 2
        UNAUTHORIZE_INSUFFICIENT_FUNDS = 3

    class Error(Exception):
        pass

    class BadRequestError(Error):
        pass

    class InvalidPaymentMethodError(Error):
        pass

    def __init__(self, default_product_id):
        self._default_product_id = default_product_id
        self._post_start_action = self.PostStartAction.DO_NOTHING

    @classmethod
    def from_settings(cls, push_client):
        return cls(
            default_product_id=cars.settings.TRUST['default_product_id'],
        )

    @classmethod
    def clear(cls):
        cls._payment_methods = collections.defaultdict(dict)
        cls._payments = collections.defaultdict(dict)
        cls._products = {}

    def set_post_start_action(self, action):
        self._post_start_action = action

    def create_payment_method(self, uid, id_, account=None):
        if account is None:
            account = '555555****4444'

        payment_method = {
            'account': account,
            'binding_systems': [
                'trust'
            ],
            'binding_ts': '1516176111.709',
            'binding_type': 'fake',
            'expiration_month': '04',
            'expiration_year': '2019',
            'expired': False,
            'holder': 'asdads asdasd',
            'id': id_,
            'orig_uid': '4008631372',
            'payment_method': 'card',
            'region_id': 225,
            'system': 'MasterCard',
        }
        self._payment_methods[uid][payment_method['id']] = payment_method
        return payment_method

    def create_product(self, name, product_id, product_type):
        assert product_id not in self._products
        product = {
            'balance_id': random.randint(1, 100000),
            'name': name,
            'partner_id': None,
            'prices': [],
            'product_id': str(product_id),
            'product_type': product_type,
            'status': 'success'
        }
        self._products[product['product_id']] = product
        return product

    def get_payment_methods(self, uid):
        return {
            'bound_payment_methods': list(self._payment_methods[uid].values()),
            'status': 'success',
        }

    def get_payment(self, uid, purchase_token):
        try:
            response = self._payments[uid][purchase_token]
        except KeyError:
            response = {
                'status': 'error',
                'status_code': 'payment_not_found'
            }
        return response

    def create_payment(self, uid, paymethod_id, amount, user_email,
                       currency='RUB', fiscal_title=None, fiscal_nds=None, back_url=None):

        try:
            product = self._products[self._default_product_id]
        except KeyError:
            return {
                'status': 'error',
                'status_code': 'product_not_found'
            }

        try:
            payment_method = self._payment_methods[uid][paymethod_id]
        except KeyError:
            raise self.InvalidPaymentMethodError

        payment = {
            'amount': str(amount),
            'orig_amount': str(amount),
            'card_type': payment_method['system'],
            'currency': currency,
            'current_amount': str(amount),
            'orders': [
                {
                    'current_qty': '0.00',
                    'order_id': '230310',
                    'order_ts': str(time.time()),
                    'orig_amount': str(amount),
                    'paid_amount': '0.00',
                    'product_id': product['product_id'],
                    'product_name': product['name'],
                    'product_type': product['product_type'],
                    'uid': str(uid),
                }
            ],
            'payment_method': payment_method['payment_method'],
            'payment_status': 'not_started',
            'payment_timeout': '1200.000',
            'paymethod_id': payment_method['id'],
            'purchase_token': uuid.uuid4().hex,
            'status': 'success',
            'uid': str(uid),
            'update_ts': time.time(),
            'user_email': user_email,
        }

        self._payments[uid][payment['purchase_token']] = payment

        return {
            'purchase_token': payment['purchase_token'],
            'status': 'success',
        }

    def start_payment(self, uid, purchase_token):
        try:
            payment = self._payments[uid][purchase_token]
        except KeyError:
            return {
                'status': 'error',
                'status_code': 'payment_not_found',
            }

        ts = str(time.time())
        payment['payment_status'] = 'started'
        payment['start_ts'] = ts
        payment['update_ts'] = ts

        if self._post_start_action is self.PostStartAction.DO_NOTHING:
            pass
        elif self._post_start_action is self.PostStartAction.AUTHORIZE:
            self.authorize_payment(uid=uid, purchase_token=purchase_token)
        elif self._post_start_action is self.PostStartAction.UNAUTHORIZE_NO_REASON:
            self.unauthorize_payment(
                uid=uid,
                purchase_token=purchase_token,
                resp_code='',
                resp_desc='',
            )
        elif self._post_start_action is self.PostStartAction.UNAUTHORIZE_INSUFFICIENT_FUNDS:
            self.unauthorize_payment(
                uid=uid,
                purchase_token=purchase_token,
                resp_code='not_enough_funds',
                resp_desc='ActionCode [116] - [not_enough_money]',
            )
        else:
            raise RuntimeError('unreachable: {}'.format(self._post_start_action))

        return payment

    def clear_payment(self, uid, purchase_token):
        payment = self._payments[uid][purchase_token]

        if payment['payment_status'] not in {'authorized', 'cleared'}:
            return {
                'status': 'error',
                'status_code': 'invalid_state'
            }

        ts = str(time.time())
        payment['update_ts'] = ts
        payment['final_status_ts'] = ts
        payment['payment_status'] = 'cleared'

        return {
            'status': 'success',
            'status_code': 'payment_is_updated'
        }

    def unhold_payment(self, uid, purchase_token):
        payment = self._payments[uid][purchase_token]

        if payment['payment_status'] not in {'authorized', 'canceled'}:
            return {
                'status': 'error',
                'status_code': 'invalid_state'
            }

        ts = str(time.time())
        payment['update_ts'] = ts
        payment['final_status_ts'] = ts
        payment['payment_status'] = 'canceled'

        return {
            'status': 'success',
            'status_code': 'payment_is_updated'
        }

    def authorize_payment(self, uid, purchase_token):
        payment = self._payments[uid][purchase_token]
        assert payment['payment_status'] == 'started'
        payment['payment_status'] = 'authorized'
        payment['update_ts'] = str(time.time())

    def unauthorize_payment(self, uid, purchase_token, resp_code, resp_desc):
        payment = self._payments[uid][purchase_token]
        assert payment['payment_status'] == 'started'
        payment['payment_status'] = 'not_authorized'
        payment['update_ts'] = str(time.time())
        payment['payment_resp_code'] = resp_code
        payment['payment_resp_desc'] = resp_desc

    def resize_payment(self, uid, purchase_token, amount, qty):
        now_ts = str(time.time())

        payment = self._payments[uid][purchase_token]
        orders = [order for order in payment['orders']]
        assert len(orders) == 1, 'can resize only single order payments'

        refund = {
            'amount': str(decimal.Decimal(payment['amount']) - amount),
            'confirm_ts': now_ts,
            'create_ts': now_ts,
            'description': 'UpdateBasket',
            'trust_refund_id': uuid.uuid1().hex,
        }
        refunds = payment.setdefault('refunds', [])
        refunds.append(refund)

        payment['current_amount'] = amount
        order = payment['orders'][0]
        order['amount'] = amount
        order['current_qty'] = qty

    def refund_payment(self, uid, purchase_token):
        now_ts = str(time.time())

        payment = self._payments[uid][purchase_token]
        assert payment['payment_status'] == 'cleared'

        payment['payment_status'] = 'refunded'
        payment['current_amount'] = '0.00'
        payment['update_ts'] = now_ts

        refund = {
            'amount': payment['orig_amount'],
            'confirm_ts': now_ts,
            'create_ts': now_ts,
            'description': 'Cancel payment',
            'trust_refund_id': uuid.uuid1().hex,
        }
        payment['refunds'] = [refund]
