import collections
import contextlib
import datetime
import importlib
import inspect
import logging
import math
import operator
import os
import re
import time
import unittest.mock

import phonenumbers
if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
    import pytz
import shapely
import yt.wrapper as yt

import cars.core.parking_areas
import cars.settings
from cars.aggregator.static_data import cities


LOGGER = logging.getLogger(__name__)


try:
    import rtree.index

    CITIES_RTREE = rtree.index.Index()
    for c in cities.CITIES.values():
        CITIES_RTREE.insert(c.code, c.location.bounds, obj=c)

    def get_city(lat, lon):
        bbox = (lon, lat, lon, lat)
        results = list(CITIES_RTREE.nearest(bbox, objects=True))
        if not results:
            return None
        city = results[0].object
        return city

except (ImportError, OSError):
    # libspatialindex_c.so required by rtree is not unavailable on YT.
    LOGGER.warning('rtree is not available. falling back to python implementation')

    def get_city(lat, lon):
        point = (lon, lat)
        city = None
        min_distance = None
        for candidate_city in cities.CITY_LIST:
            distance = euclidian_distance(point, candidate_city.location.coords[0])
            if min_distance is None or distance < min_distance:
                city = candidate_city
                min_distance = distance
        return city


Location = collections.namedtuple('Location', ['lat', 'lon'])


class DateTimeHelper(object):
    if os.environ.get('DJANGO_SETTINGS_MODULE') is not None:
        DEFAULT_TIMEZONE = pytz.timezone('Europe/Moscow')

    def change_timezone(self, value):
        """convert datetime to a specific timezone"""
        return self.DEFAULT_TIMEZONE.normalize(value.astimezone(self.DEFAULT_TIMEZONE))

    def localize(self, value):
        """replace timezone with a default one"""
        return self.DEFAULT_TIMEZONE.localize(value)

    def utc_localize(self, value):
        """set timezone to UTC"""
        return pytz.UTC.localize(value)

    def timestamp_to_datetime(self, value, truncate=False, is_local=False):
        """convert timestamp to a local datetime

        timestamp is supposed to represent UTC time

        is_local indicates that timestamp is given in a default timezone
        """
        value = float(value) if not truncate else int(value)
        datetime_value = datetime.datetime.utcfromtimestamp(value)

        if is_local:
            datetime_value = self.localize(datetime_value)
        else:
            datetime_value = self.utc_localize(datetime_value)

        normalized_datetime_value = self.change_timezone(datetime_value)
        return normalized_datetime_value

    def timestamp_ms_to_datetime(self, value, is_local=False):
        """convert timestamp given in ms to a local datetime"""
        return self.timestamp_to_datetime(int(value) / 1000, truncate=False, is_local=is_local)

    def datetime_to_timestamp(self, value, truncate=False):
        result = self.change_timezone(value).timestamp()
        return result if not truncate else int(result)

    def now(self):
        return self.change_timezone(self.utc_now())

    def utc_now(self):
        datetime_value = self.utc_localize(datetime.datetime.utcnow())
        return datetime_value

    def timestamp_now(self, truncate=False):
        result = time.time()
        return result if not truncate else int(result)

    def duration_to_str(self, duration_s, truncate=True):
        if duration_s is None:
            return ''

        duration_s = int(duration_s) if truncate else duration_s
        return str(datetime.timedelta(seconds=duration_s))

    def add_years(self, d, years):
        """
        https://stackoverflow.com/a/15743908

        Return a date that's `years` years after the date (or datetime)
        object `d`. Return the same calendar date (month and day) in the
        destination year, if it exists, otherwise use the following day
        (thus changing February 29 to March 1).
        """
        try:
            new_year = d.year + years
            new_date = d.replace(year=new_year)
        except ValueError:
            new_date = d + (datetime.date(d.year + years, 1, 1) - datetime.date(d.year, 1, 1))
        return new_date


datetime_helper = DateTimeHelper()


class PhoneNumberHelper(object):
    VALID_PHONE_RE = re.compile('\+?[0-9()\- ]{10,}')  # simplified regexp

    DEFAULT_REGION = 'RU'

    def normalize_phone_number(self, phone_number, *, default=None):
        if phone_number is None:
            return default

        if len(phone_number) == 10:  # local Russia without any codes
            normalized_result = self.normalize_phone_number('+7' + phone_number)  # use 'None' default
            if normalized_result is not None:
                return normalized_result

        normalized_result = default

        if phone_number.startswith(('9810', '9007')):  # international from Russia and Turkey or local Russia
            prefix_len = 4
            is_international = (len(phone_number) != 14)  # local: '9xxx', 3-digit code and 7-digit phone number
        elif phone_number.startswith('98'):  # international or local Russia
            prefix_len = 2
            is_international = (len(phone_number) != 12)  # local: '98', 3-digit code and 7-digit phone number
        else:
            prefix_len = 0
            is_international = None

        if prefix_len:
            phone_number = phone_number[prefix_len:]

        if is_international:
            phone_number = '+' + phone_number

        try:
            phone_number_instance = phonenumbers.parse(phone_number or '', region=self.DEFAULT_REGION)
        except phonenumbers.NumberParseException:
            pass
        else:
            if phonenumbers.is_valid_number(phone_number_instance):
                normalized_result = phonenumbers.format_number(
                    phone_number_instance, phonenumbers.PhoneNumberFormat.E164
                )

        return normalized_result

    def iter_all_possible_phone_numbers(self, text):
        regexp_phones = self.VALID_PHONE_RE.findall(text)
        normalized_possible_phones = (self.normalize_phone_number(p) for p in regexp_phones)
        yield from {p for p in normalized_possible_phones if p}


phone_number_helper = PhoneNumberHelper()


def euclidian_distance(x, y):
    dx = x[0] - y[0]
    dy = x[1] - y[1]
    return math.sqrt(dx ** 2 + dy ** 2)


def make_yt_client(section):

    def get_additional_files_to_archive():
        files = []

        additional_dirs = []  # List or (src_dir, dst_dir) tuples.

        # Collect Shapely vendored geos library.
        shapely_libs_src_dir = os.path.join(os.path.dirname(shapely.__file__), '.libs')
        shapely_libs_dst_dir = os.path.join('shapely', '.libs')
        additional_dirs.append((shapely_libs_src_dir, shapely_libs_dst_dir))

        # Collect parking areas .wkt files.
        parking_areas_path = inspect.getfile(cars.core.parking_areas)
        parking_areas_static_src_dir = os.path.join(os.path.dirname(parking_areas_path), 'static')
        parking_areas_static_dst_dir = os.path.join('cars', 'core', 'parking_areas', 'static')
        additional_dirs.append((parking_areas_static_src_dir, parking_areas_static_dst_dir))

        for src_dir, dst_dir in additional_dirs:
            try:
                filenames = os.listdir(src_dir)
            except FileNotFoundError:
                LOGGER.warning(
                    '%s library directory not found. YT operations will fail.',
                    src_dir,
                )
                continue

            for filename in filenames:
                src_path = os.path.join(src_dir, filename)
                dst_path = os.path.join(dst_dir, filename)
                files.append((src_path, dst_path))

        return files

    # Pass all CARS_* environment variables into operations.
    operation_environment = {}
    for env_var_name, env_var_value in os.environ.items():
        if env_var_name.startswith('CARS_') or env_var_name.startswith('CARSHARING_'):
            operation_environment[env_var_name] = env_var_value

    YT = cars.settings.YT[section]
    client = yt.client.Yt(proxy=YT['proxy'], token=YT['token'])
    client.config['pickling']['python_binary'] = 'python3.4'
    client.config['pickling']['ignore_yson_bindings_for_incompatible_platforms'] = False
    client.config['pickling']['additional_files_to_archive'] = get_additional_files_to_archive()
    client.config['pickling']['force_using_py_instead_of_pyc'] = True
    client.config['spec_overrides'] = {
        'mapper': {
            'environment': operation_environment,
        },
        'reducer': {
            'environment': operation_environment,
        },
    }

    return client


def ensure_yt_dir(dirname, client):
    if not client.exists(dirname):
        client.create('map_node', dirname, recursive=True)


def import_class(path):
    module_path, class_name = path.rsplit('.', 1)
    module = importlib.import_module(module_path)
    class_ = getattr(module, class_name)

    if issubclass(class_, unittest.mock.Mock):
        # mock class cannot be used directly, an instance must be created
        return class_()

    return class_


def _attrs_to_getter(key, getter):
    if isinstance(key, str):
        key = (key,)

    if isinstance(key, (tuple, list)) and all(isinstance(x, str) for x in key):
        key = getter(*key)

    return key


def collection_to_mapping(collection, *, key=None, attr_key=None, item_key=None, value=None, multiple_values=True):
    assert isinstance(collection, collections.Iterable)
    assert len([x for x in (key, attr_key, item_key) if x is not None]) == 1

    target_collection = collections.OrderedDict()

    if attr_key is not None:
        key = _attrs_to_getter(attr_key, operator.attrgetter)
    elif item_key is not None:
        key = _attrs_to_getter(item_key, operator.itemgetter)
    else:
        assert isinstance(key, collections.Callable)

    for item in collection:
        item_key = key(item)
        item_value = item if value is None else value(item)

        if multiple_values:
            target_collection.setdefault(item_key, [])
            target_collection[item_key].append(item_value)
        else:
            target_collection[item_key] = item_value

    return target_collection


@contextlib.contextmanager
def handle_error(*, re_raise=True, exc_type=Exception):
    try:
        yield
    except exc_type as exc:
        LOGGER.exception('a handled exception occurred: {}'.format(str(exc)))
        if re_raise:
            raise
