from rest_framework.response import Response
from rest_framework.status import HTTP_404_NOT_FOUND

import cars.settings
from cars.carsharing.models import Car
from cars.carsharing.models.tag import CarTag
from .base import DeptransAPIView
from ..serializers import (
    DeptransCarSerializer,
    DeptransCarInfoViewArgumentsSerializer,
)


class DeptransCarInfoView(DeptransAPIView):

    arguments_serializer_class = DeptransCarInfoViewArgumentsSerializer
    serializer_class = DeptransCarSerializer

    def get_queryset(self):
        return (
            Car.objects.using(cars.settings.DB_RO_ID)
        )

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        car = (
            self.get_queryset()
            .select_related('model')
            .filter(imei=request.arguments['id'])
            .first()
        )

        if car is None or car.model_id in self.banned_model_codes:
            return Response(
                status=HTTP_404_NOT_FOUND
            )

        car_tags = (
            CarTag.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                object_id=car.id,
                tag__in=[
                    'parking_free',
                    'peterburg_tag',
                    'kazan_tag',
                ]
            )
        )
        if car_tags.count() != 1 or car_tags.first().tag != 'parking_free':
            return Response(
                status=HTTP_404_NOT_FOUND
            )

        return Response(self.serializer_class(car).data)


class DeptransCarListView(DeptransAPIView):

    BANNED_IDS = [
        'c5d2ccb3-22c4-4dd6-be8a-6eab5c744a53',
        '5367dd67-cdb9-4811-b101-8a67fae88b69',
        'fd604491-9077-4716-bf8f-392adeab6e45',
    ]

    def do_get(self, request, *args, **kwargs):  # pylint: disable=unused-argument
        tags = (
            CarTag.objects
            .using(cars.settings.DB_RO_ID)
            .filter(
                tag__in=[
                    'parking_free',
                    'peterburg_tag',
                    'kazan_tag',
                ]
            )
        )

        car_ids = set()
        for tag in tags:
            if tag.tag == 'parking_free':
                car_ids.add(tag.object_id)

        for tag in tags:
            if tag.tag != 'parking_free' and tag.object_id in car_ids:
                car_ids.remove(tag.object_id)

        qs = (
            Car.objects
            .using(cars.settings.DB_RO_ID)
            .filter(id__in=car_ids)
            .exclude(id__in=self.BANNED_IDS)
            .only('number', 'imei', 'model')
            .select_related('model')
        )

        imei_numbers = [
            DeptransCarSerializer(x).data for x in qs
        ]

        print(len(imei_numbers))

        return Response(
            imei_numbers
        )
