import logging
import re
import time

from django.urls import resolve, Resolver404
from django_yauth.util import get_real_ip
from rest_framework.views import APIView
from django.http import HttpResponse
from rest_framework.status import HTTP_401_UNAUTHORIZED, HTTP_403_FORBIDDEN

import cars.settings
from cars.core.solomon import SOLOMON
from cars.core.tvm2 import TVM2ServiceTicket
from cars.core.util import Location


LOGGER = logging.getLogger(__name__)


class IDMxTVM2Middleware(object):
    '''Authenticate IDM requests by TVM2'''

    def __init__(self, get_response):
        self.get_response = get_response

        self._idm_client_id = cars.settings.IDM['client_tvm_id']
        self._ticket_parser = TVM2ServiceTicket(
            host=cars.settings.USERS['documents']['datasync']['tvm2']['host'],
            source=cars.settings.USERS['documents']['datasync']['tvm2']['source'],
            destination=None,
            secret=None,
        )

    def __call__(self, request):
        if request.path.lstrip('/').startswith('api/idm/v1'):
            ticket = request.META.get('HTTP_X_YA_SERVICE_TICKET')
            if not ticket:
                response = HttpResponse({'error': 'no tvm ticket'}, status=HTTP_401_UNAUTHORIZED)
                return response
            elif not self._ticket_parser.authenticate_ticket(self._idm_client_id, ticket):
                response = HttpResponse({'error': 'bad tvm ticket'}, status=HTTP_403_FORBIDDEN)
                return response

        return self.get_response(request)


class RequestMetaSetterMiddleware(object):
    '''Set request metadata'''

    def __init__(self, get_response):
        self.get_response = get_response

    def __call__(self, request):
        self._set_request_meta(request)
        response = self.get_response(request)
        return response

    def _set_request_meta(self, request):
        request.location = self._get_location(request)
        request.oauth_token = self._parse_oauth_token(request)
        request.user_agent = request.META.get('HTTP_USER_AGENT')
        request.user_ip = get_real_ip(request)

    def _get_location(self, request):
        lat = request.META.get('HTTP_LAT')
        lon = request.META.get('HTTP_LON')

        if not (lat and lon):
            return None

        try:
            lat = float(lat)
            lon = float(lon)
        except ValueError:
            return None

        location = Location(lat=lat, lon=lon)

        return location

    def _parse_oauth_token(self, request):
        auth = request.META.get('HTTP_AUTHORIZATION')
        if not auth:
            return None

        match = re.match(r'(?:OAuth|Bearer) (?P<token>.+)', auth)
        if not match:
            return None

        return match.group('token')


class SolomonMonitoringMiddleware(object):

    undefined_label_value = 'undefined'

    def __init__(self, get_response):
        self.get_response = get_response

    def _get_now_ms(self):
        return time.time() * 1000

    def __call__(self, request):
        start_time_ms = self._get_now_ms()
        response = self.get_response(request)
        response_time = self._get_now_ms() - start_time_ms

        url_name = self._resolve_url_name(request)

        self._monitor_response_time(
            method=request.method,
            url_name=url_name,
            response_status_code=response.status_code,
            response_time=response_time,
        )

        return response

    def _resolve_url_name(self, request):
        try:
            resolved = resolve(request.path)
            url_name = resolved.view_name
        except Resolver404:
            url_name = ''
        except Exception:
            LOGGER.exception('failed to resolve request url: %s', request.path)
            url_name = ''
        return url_name

    def _monitor_response_time(self, method, url_name, response_status_code, response_time):
        labels = {
            'method': method.upper(),
            'url-name': url_name,
            'status': response_status_code,
        }
        SOLOMON.set_value('api.response.time', response_time, labels=labels)
