import datetime
import time

from django.utils import timezone, translation
from rest_framework import serializers

from cars.core.util import phone_number_helper


class BaseSerializer(serializers.Serializer):

    def create(self, *args, **kwargs):  # pylint: disable=arguments-differ,useless-super-delegation
        '''Fix pylint abstract-method complaints'''
        return super().create(*args, **kwargs)

    def update(self, *args, **kwargs):  # pylint: disable=arguments-differ,useless-super-delegation
        '''Fix pylint abstract-method complaints'''
        return super().update(*args, **kwargs)


class TimestampField(serializers.FloatField):

    def to_representation(self, obj):
        if obj:
            return obj.timestamp()
        return None

    def to_internal_value(self, data):
        # The last microsecond may be lost due to rounding.
        # Hack with replacing microseconds fixes it.
        timestamp = super().to_internal_value(data)

        if timestamp < 0:
            timestamp = time.time() + timestamp

        date_coarse = datetime.datetime.utcfromtimestamp(timestamp)
        date_precise = date_coarse.replace(microsecond=int(timestamp * 1000000 % 1000000))
        date_aware = timezone.utc.localize(date_precise)

        return date_aware


class PhoneNumberField(serializers.CharField):
    default_error_messages = {
        'invalid': translation.ugettext_lazy('"{value}" is not a valid phone number.'),
    }

    def to_internal_value(self, data):
        if isinstance(data, str):
            value = phone_number_helper.normalize_phone_number(data)
            if value is not None:
                return value

        self.fail('invalid', value=data)

    def to_representation(self, value):
        return str(value) if value is not None else None
