import enum
import uuid

from rest_framework.exceptions import NotFound


def parse_uuid_or_404(uuid_str):
    try:
        return uuid.UUID(uuid_str)
    except ValueError:
        raise NotFound


class ReqAnsLogger(object):

    class Policy(enum.Enum):
        NONE = 0
        META = 1
        FULL = 2

    def __init__(self, push_client):
        self._push_client = push_client

    def log(self, request, response, policy):
        if policy is self.Policy.NONE:
            return

        data = {
            'user': self._get_user_data(request),
            'request': {
                'meta': self._get_request_meta(request),
                'method': request.method,
                'query_params': request.query_params.dict(),
            },
            'response': {
                'status_code': response.status_code,
            },
        }

        if policy is self.Policy.FULL:
            data['request']['data'] = request.data if isinstance(request.data, dict) else None
            data['response']['data'] = response.data if isinstance(response.data, dict) else None

        self._push_client.log(type_='reqans', data=data)

    def _get_request_meta(self, request):
        meta = {}
        for key, value in request.META.items():
            if not isinstance(value, str):
                continue
            if key in ['HTTP_AUTHORIZATION', 'HTTP_COOKIE']:
                continue
            meta[key] = value
        return meta

    def _get_user_data(self, request):
        if not request.user.is_authenticated:
            return None

        data = {
            'id': str(request.user.id),
            'uid': request.user.uid,
        }

        return data
