import logging

from django.http.response import StreamingHttpResponse

import rest_framework.views
from rest_framework.exceptions import ParseError
from rest_framework.generics import GenericAPIView
from rest_framework.response import Response
from rest_framework.status import HTTP_405_METHOD_NOT_ALLOWED, HTTP_410_GONE

import cars.settings
from cars.core.push_client import CarsharingPushClientLogger
from .lock import UwsgiLock
from .serializers import BaseSerializer
from .util import ReqAnsLogger


LOGGER = logging.getLogger(__name__)


class CarsharingAPIView(GenericAPIView):

    arguments_serializer_class = None
    serializer_class = BaseSerializer

    reqans_logger = ReqAnsLogger(
        push_client=CarsharingPushClientLogger(
            filename=cars.settings.REQANS_LOGGER['push_client']['filename'],
            lock=UwsgiLock(),
        ),
    )

    # Can be fine-tuned in get_reqans_logger_policy method.
    reqans_logger_policy = ReqAnsLogger.Policy.META

    def dispatch(self, request, *args, **kwargs):
        response = super().dispatch(request, *args, **kwargs)

        try:
            self.reqans_logger.log(
                request=self.request,
                response=response,
                policy=self.get_reqans_logger_policy(request=request),
            )
        except Exception:
            LOGGER.exception('reqans logger failed')

        return response

    def delete(self, request, *args, **kwargs):
        if not hasattr(self, 'do_delete'):
            return Response(status=HTTP_405_METHOD_NOT_ALLOWED)
        request.arguments = self._parse_arguments(
            request=request,
            data=request.data,
        )
        return self.do_delete(request, *args, **kwargs)

    def get(self, request, *args, **kwargs):
        if not hasattr(self, 'do_get'):
            return Response(status=HTTP_405_METHOD_NOT_ALLOWED)
        request.arguments = self._parse_arguments(
            request=request,
            data=request.query_params,
        )
        return self.do_get(request, *args, **kwargs)

    def post(self, request, *args, **kwargs):
        if not hasattr(self, 'do_post'):
            return Response(status=HTTP_405_METHOD_NOT_ALLOWED)
        request.arguments = self._parse_arguments(
            request=request,
            data=request.data,
        )
        return self.do_post(request, *args, **kwargs)

    def put(self, request, *args, **kwargs):
        if not hasattr(self, 'do_put'):
            return Response(status=HTTP_405_METHOD_NOT_ALLOWED)
        request.arguments = self._parse_arguments(
            request=request,
            data=request.data,
        )
        return self.do_put(request, *args, **kwargs)

    def get_reqans_logger_policy(self, request):  # pylint: disable=unused-argument
        return self.reqans_logger_policy

    def get_arguments_serializer_class(self):
        return self.arguments_serializer_class

    def get_arguments_serializer(self, *args, **kwargs):
        arguments_serializer_class = self.get_arguments_serializer_class()

        if arguments_serializer_class is None:
            return None

        kwargs['context'] = self.get_arguments_serializer_context()

        return arguments_serializer_class(*args, **kwargs)

    def get_arguments_serializer_context(self):
        return {
            'request': self.request,
        }

    def _parse_arguments(self, request, data):
        serializer = self.get_arguments_serializer(data=data)

        if serializer is None:
            return None

        if not serializer.is_valid():
            data = {
                'errors': serializer.errors,
            }
            raise ParseError(detail=data)

        return serializer.validated_data

    def get_authenticate_header(self, request):
        return 'OAuth'


class PingView(CarsharingAPIView):

    reqans_logger_policy = ReqAnsLogger.Policy.NONE

    def do_get(self, _):
        return Response()


class GoneView(CarsharingAPIView):
    def get(self, request, *args, **kwargs):
        return Response(status=HTTP_410_GONE)

    def post(self, request, *args, **kwargs):
        return Response(status=HTTP_410_GONE)

    def put(self, request, *args, **kwargs):
        return Response(status=HTTP_410_GONE)

    def patch(self, request, *args, **kwargs):
        return Response(status=HTTP_410_GONE)

    def delete(self, request, *args, **kwargs):
        return Response(status=HTTP_410_GONE)


class StreamingResponseViewMixin(object):

    def get_response(self, stream, content_type, content_length):
        content_range = self.request.META.get('HTTP_RANGE')
        if content_range:
            offset, limit = content_range.split('=')[1].split('-')
            offset = int(offset)
            limit = int(limit) if limit else content_length - 1
        else:
            offset = limit = None

        response = StreamingHttpResponse(
            streaming_content=StreamIterator(stream, offset=offset, limit=limit),
            content_type=content_type,
            status=200,
        )
        response['Accept-Ranges'] = 'bytes'
        response['Content-Length'] = content_length

        if offset is not None:
            response['Content-Length'] = limit - offset + 1
            response['Content-Range'] = 'bytes {}-{}/{}'.format(offset, limit, content_length)
            response.status_code = 206

        return response


class StreamIterator(object):

    def __init__(self, fileobj, offset=None, limit=None, chunk_size=8096):
        self._fileobj = fileobj
        self._offset = offset
        self._limit = limit
        self._chunk_size = chunk_size

    def __iter__(self):
        read_bytes = 0

        if self._offset:
            data = self._fileobj.read(self._offset)
            read_bytes += len(data)

        while True:
            if self._limit:
                chunk_size = min(self._chunk_size, self._limit - read_bytes + 1)
            else:
                chunk_size = self._chunk_size
            if chunk_size == 0:
                break

            data = self._fileobj.read(chunk_size)
            read_bytes += len(data)
            if not data:
                break

            yield data


def exception_handler(exc, context):
    return rest_framework.views.exception_handler(exc=exc, context=context)
